package com.ihaoin.hooloo.device.view;

import android.content.Context;
import android.graphics.Typeface;
import android.os.Message;
import android.util.AttributeSet;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.view.animation.Animation;
import android.view.animation.AnimationUtils;
import android.view.animation.BounceInterpolator;
import android.widget.ListView;
import android.widget.RelativeLayout;
import android.widget.TextView;

import androidx.annotation.NonNull;

import com.fasterxml.jackson.databind.JsonNode;
import com.ihaoin.hooloo.device.HLApplication;
import com.ihaoin.hooloo.device.R;
import com.ihaoin.hooloo.device.adapter.TrolleyAdapter;
import com.ihaoin.hooloo.device.component.NetworkHandler;
import com.ihaoin.hooloo.device.component.Rotate3dAnimation;
import com.ihaoin.hooloo.device.config.Base;
import com.ihaoin.hooloo.device.data.TrolleyGoods;
import com.ihaoin.hooloo.device.data.enums.SkuState;
import com.ihaoin.hooloo.device.data.vo.Goods;
import com.ihaoin.hooloo.device.data.vo.Sku;
import com.ihaoin.hooloo.device.network.HttpUtil;
import com.ihaoin.hooloo.device.util.JsonUtils;
import com.ihaoin.hooloo.device.util.RandomUtils;
import com.ihaoin.hooloo.device.util.Utils;

import java.math.BigDecimal;
import java.util.Optional;
import java.util.stream.Collectors;

public class TrolleyView extends RelativeLayout {
    private Boolean state = false;
    private Boolean trolleyAniming = false;

    private View butnClear;
    private View viewBackground;
    private View layoutTrolley;
    private ListView listTrolley;
    private View layoutSettleBar;
    private TextView txtCount;
    private TextView txtTotal;
    private View butnPay;

    private TrolleyAdapter trolleyAdapter;

    private void init() {
        View view = LayoutInflater.from(getContext()).inflate(R.layout.view_trolley, this);

        butnClear = view.findViewById(R.id.butn_clear);
        viewBackground = view.findViewById(R.id.view_background);
        layoutTrolley = view.findViewById(R.id.layout_trolley);
        listTrolley = view.findViewById(R.id.list_trolley);
        layoutSettleBar = view.findViewById(R.id.layout_settle_bar);
        txtCount = view.findViewById(R.id.txt_count);
        txtTotal = view.findViewById(R.id.txt_total);
        butnPay = view.findViewById(R.id.butn_pay);
        this.setView();

        trolleyAdapter = new TrolleyAdapter(this.getContext(), this);
        listTrolley.setAdapter(trolleyAdapter);

        butnClear.setOnClickListener(v -> clearTrolley());
        layoutSettleBar.setOnClickListener(v -> toggleTrolley());
        viewBackground.setOnClickListener(v -> closeTrolley());

        butnPay.setOnClickListener(v -> showPayDialog());

        Typeface tfHalf = Utils.getHalfFont(getContext());
        txtCount.setTypeface(tfHalf);
        txtTotal.setTypeface(tfHalf);
        txtCount.getPaint().setFakeBoldText(true);
        txtTotal.getPaint().setFakeBoldText(true);
    }

    public void prepareAddGoods(Goods goods, Sku sku, Integer count) {
        CheckGoodsHandler handler = new CheckGoodsHandler(goods, sku, count);
        HttpUtil.checkSku(sku.getSkuId(), handler);
    }

    class CheckGoodsHandler extends NetworkHandler {
        private Goods goods;
        private Sku sku;
        private Integer count;

        public CheckGoodsHandler(Goods goods, Sku sku, Integer count) {
            super();
            this.goods = goods;
            this.sku = sku;
            this.count = count;
        }

        @Override
        public void handleMessage(@NonNull Message msg) {
            super.handleMessage(msg);
            Object obj = msg.obj;
            if (msg.what != 1 || obj == null) {
                Utils.showToast(getContext(), R.string.error_check_sku_wrong);
                return;
            }

            JsonNode rsp = JsonUtils.readValue(obj, JsonNode.class);
            if (!SkuState.SELLIN.getCode().equals(rsp.get("state").asInt())) {
                Utils.showToast(getContext(), R.string.error_check_sku);
                return;
            }
            addGoods(goods, sku, count);
        }
    }

    /** 添加商品到购物车 - 商品详情、商品列表 */
    public void addGoods(Goods goods, Sku sku, Integer count) {
        if (HLApplication.getGoodsCount() >= HLApplication.getCountOfOrder()) {
            Utils.showToast(this.getContext(), getContext().getString(R.string.count_limit, HLApplication.getCountOfOrder().toString()));
            return;
        }
        Optional<TrolleyGoods> optional = HLApplication.getTrolleyGoods().stream().filter(x -> x.getGoods().getGoodsId().equals(goods.getGoodsId()) && x.getSku().getSkuId().equals(sku.getSkuId())).findAny();
        // 判断是否在购物车里存在同样规格的产品
        if (optional.isPresent()) {
            TrolleyGoods g = optional.get();
            g.setCount(g.getCount() + count);
        } else {
            TrolleyGoods g = new TrolleyGoods();
            g.setId(RandomUtils.nextLong(HLApplication.getTrolleyGoods().stream().map(TrolleyGoods::getId).collect(Collectors.toList())));
            g.setChecked(true);
            g.setGoods(goods);
            g.setSku(sku);
//            g.setGoodId(goods.getGoodsId());
//            g.setName(goods.getName());
//            g.setDesc(goods.getDesc());
//            g.setPics(goods.getPics());
//            g.setTags(goods.getTags());
//            g.setSkuId(sku.getSkuId());
//            g.setState(sku.getState());
//            g.setIsDefault(sku.getIsDefault());
//            g.setPrice(sku.getPrice());
//            g.setDiscount(sku.getDiscount());
//            g.setRules(sku.getRules());
            g.setCount(count);
            HLApplication.getTrolleyGoods().add(g);
        }

        this.goodsChanged();
    }

    /** 购物车商品增加数量 */
    public void addGoods(Integer position) {
        if (HLApplication.getGoodsCount() >= HLApplication.getCountOfOrder()) {
            Utils.showToast(this.getContext(), getContext().getString(R.string.count_limit, HLApplication.getCountOfOrder().toString()));
            return;
        }
        TrolleyGoods goods = HLApplication.getTrolleyGoods().get(position);
        goods.setCount(goods.getCount() + 1);
        this.goodsChanged();
    }

    /** 购物车商品减少数量 */
    public void subtractGoods(Integer position) {
        TrolleyGoods goods = HLApplication.getTrolleyGoods().get(position);
        if (goods.getCount() <= 1) {
            HLApplication.getTrolleyGoods().remove(goods);
        } else {
            goods.setCount(goods.getCount() - 1);
        }
        this.goodsChanged();
    }

    public void goodsChanged() {
        this.trolleyAdapter.notifyDataSetChanged();
        this.txtCount.setText(HLApplication.getGoodsCheckedCount().toString());
        this.txtTotal.setText(Utils.toString(HLApplication.getGoodsCheckedTotal()));
        this.setView();
        this.measureListHeight();

        float centerX = txtCount.getWidth() / 2f;
        float centerY = txtCount.getHeight() / 2f;

        Animation anim = new Rotate3dAnimation(180f, 360f, centerX, centerY, 0f, Rotate3dAnimation.ROTATE_Y_AXIS, true);
        anim.setFillAfter(true); // 设置保持动画最后的状态
        anim.setDuration(800); // 设置动画时间
        anim.setInterpolator(new BounceInterpolator()); // 设置插入器
        txtCount.startAnimation(anim);
    }

    public Boolean checkGoodsCount() {
        if (HLApplication.getGoodsCount() > HLApplication.getCountOfOrder()) {
            Utils.showToast(this.getContext(), getContext().getString(R.string.count_limit, HLApplication.getCountOfOrder().toString()));
            return false;
        }
        return true;
    }

    private void setView() {
        if (HLApplication.getTrolleyGoods().isEmpty()) {
            this.setVisibility(View.INVISIBLE);
            viewBackground.setVisibility(View.INVISIBLE);
            layoutTrolley.setVisibility(View.INVISIBLE);
            state = false;
            Utils.fastBroadcast(getContext(), Base.BROADCAST_ACTION_TROLLEY_HIDE);
        } else {
            this.setVisibility(View.VISIBLE);
            Utils.fastBroadcast(getContext(), Base.BROADCAST_ACTION_TROLLEY_SHOW);
        }
    }

    private void measureListHeight() {
        int count = HLApplication.getTrolleyGoods().size() > 3 ? 3 : HLApplication.getTrolleyGoods().size();
        int height = Utils.getDimens(this.getContext(), R.dimen.trolley_item_height) * count;
        ViewGroup.LayoutParams layoutParams = listTrolley.getLayoutParams();
        layoutParams.height = height;
        listTrolley.setLayoutParams(layoutParams);
    }

    public void clearTrolley() {
        HLApplication.getTrolleyGoods().clear();
        setView();
    }

    private void toggleTrolley() {
        if (state) {
            closeTrolley();
        } else {
            openTrolley();
        }
    }


    private void openTrolley() {
        if (trolleyAniming) {
            return;
        }
        trolleyAniming = true;
        viewBackground.setOnClickListener(v -> closeTrolley());
        layoutTrolley.setOnClickListener(v -> {
        });
        viewBackground.setVisibility(View.VISIBLE);
        layoutTrolley.setVisibility(View.VISIBLE);

        Animation animation = AnimationUtils.loadAnimation(this.getContext(), R.anim.trolley_translate_in);
        animation.setAnimationListener(new Animation.AnimationListener() {
            @Override
            public void onAnimationStart(Animation animation) {

            }

            @Override
            public void onAnimationEnd(Animation animation) {
                trolleyAniming = false;
            }

            @Override
            public void onAnimationRepeat(Animation animation) {

            }
        });
        layoutTrolley.startAnimation(animation);
        state = true;
    }

    private void closeTrolley() {
        if (trolleyAniming) {
            return;
        }
        trolleyAniming = true;
        Animation animation = AnimationUtils.loadAnimation(this.getContext(), R.anim.trolley_translate_out);
        layoutTrolley.startAnimation(animation);
        animation.setAnimationListener(new Animation.AnimationListener() {
            @Override
            public void onAnimationStart(Animation animation) {
            }

            @Override
            public void onAnimationEnd(Animation animation) {
                viewBackground.setVisibility(View.INVISIBLE);
                layoutTrolley.setVisibility(View.INVISIBLE);
                state = false;
                trolleyAniming = false;
            }

            @Override
            public void onAnimationRepeat(Animation animation) {
            }
        });
    }

    private void showPayDialog() {
        if (HLApplication.getGoodsCheckedCount() <= 0) {
            Utils.showToast(getContext(), R.string.error_choose_goods);
            return;
        }
        if (HLApplication.getGoodsCheckedTotal().compareTo(BigDecimal.ZERO) < 0) {
            Utils.showToast(getContext(), R.string.error_price);
            return;
        }

        Utils.showConfirmOrder(getContext(), HLApplication.getTrolleyGoods());
    }

    public TrolleyView(Context context) {
        super(context);
        init();
    }

    public TrolleyView(Context context, AttributeSet attrs) {
        super(context, attrs);
        init();
    }

    public TrolleyView(Context context, AttributeSet attrs, int defStyleAttr) {
        super(context, attrs, defStyleAttr);
        init();
    }

    public TrolleyView(Context context, AttributeSet attrs, int defStyleAttr, int defStyleRes) {
        super(context, attrs, defStyleAttr, defStyleRes);
        init();
    }
}
