package com.soss.system.service.impl;

import com.soss.common.core.domain.model.LoginUser;
import com.soss.common.enums.ShopState;
import com.soss.system.domain.Order;
import com.soss.system.domain.Shop;
import com.soss.system.mapper.ShopMapper;
import com.soss.system.utils.DistanceUtil;
import org.apache.commons.lang3.BooleanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.Arrays;
import java.util.Comparator;
import java.util.List;

@Service
public class AppServiceImpl {
    @Autowired
    private ShopMapper shopMapper;
    @Autowired
    private ShopServiceImpl shopService;
    @Autowired
    private OrderServiceImpl orderService;

    public Shop getShop(LoginUser loginUser) {
        if (loginUser == null) {
            // 未登录，返回默认店铺
            return getDefaultShop();
        } else {
            // 已登录
            List<Order> myOrder = orderService.getMyOrder(loginUser.getOpenId());
            if (CollectionUtils.isEmpty(myOrder)) {
                // 未下过单返回默认店铺
                return getDefaultShop();
            }
            // 返回最后下单店铺
            Order order = myOrder.get(0);
            Shop shop = shopService.selectShopById(order.getShopId());
            return shop == null ? new Shop() : shop;
        }
    }

    public Shop getDefaultShop() {
        Shop shop = new Shop();
        shop.setIsDefault(1L);
        List<Shop> shops = shopMapper.selectShopList(shop);
        return CollectionUtils.isEmpty(shops) ? new Shop() : shops.get(0);
    }

    /**
     * 授权了地理位置的用户都展示距离最近的营业中的店铺；
     * 未授权地理位置的用户都展示默认店铺，不管店铺是否营业中；
     */
    public Shop getShop(String lng, String lat, Boolean testFlag) {
        Shop shop = new Shop();

        Comparator<Shop> shopComparator = Comparator.comparing(Shop::getRealDistance);
        if (BooleanUtils.isTrue(testFlag)) {
            shop.setStates(Arrays.asList(ShopState.TESTING.getState(), ShopState.OPEN.getState()));
        } else {
            shop.setStates(Arrays.asList(ShopState.OPEN.getState(), ShopState.CLOSE.getState()));
            shopComparator = Comparator.comparing(Shop::getState).thenComparing(Shop::getRealDistance);
        }

        List<Shop> shops = shopMapper.selectShopList(shop);
        if (CollectionUtils.isEmpty(shops)) {
            return new Shop();
        }

        shops.forEach(sp -> {
            double realDistance = DistanceUtil.getRealDistance(Double.parseDouble(lng), Double.parseDouble(lat), Double.parseDouble(sp.getLng()), Double.parseDouble(sp.getLat()));
            sp.setRealDistance(realDistance);
            String distanceDesc = DistanceUtil.getDistanceDesc(realDistance);
            sp.setDistance(distanceDesc);
        });
        shops.sort(shopComparator);
        return shops.get(0);
    }
}
