package com.soss.system.service.impl;

import com.soss.common.enums.CouponCategoryType;
import com.soss.common.enums.CouponState;
import com.soss.common.enums.CouponUserType;
import com.soss.common.enums.SkuDeleteState;
import com.soss.common.exception.ServiceException;
import com.soss.common.utils.DateUtils;
import com.soss.common.utils.StringUtils;
import com.soss.system.domain.*;
import com.soss.system.domain.vo.CouponUserVo;
import com.soss.system.domain.vo.CouponVo;
import com.soss.system.mapper.*;
import com.soss.system.service.ICouponUserService;
import com.soss.system.utils.ArrayUtil;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.time.LocalDateTime;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * <p>
 * 用户领取优惠券记录表 服务实现类
 * </p>
 *
 * @author caiyt
 * @since 2022-07-21
 */
@Service
@Slf4j
public class CouponUserServiceImpl implements ICouponUserService {
    @Autowired
    private CouponUserMapper couponUserMapper;
    @Autowired
    private CouponMapper couponMapper;
    @Autowired
    private CouponRuleMapper couponRuleMapper;
    @Autowired
    private CustomerMapper customerMapper;
    @Autowired
    private ShopMapper shopMapper;
    @Autowired
    private GoodsMapper goodsMapper;
    @Autowired
    private GoodsSkuMapper goodsSkuMapper;

    @Override
    public void giveUserCoupon(String custId, Integer couponId) {
        Customer cust = customerMapper.selectCustomerById(custId);
        Assert.notNull(cust, "用户不存在[id=" + custId + "]");
        Coupon coupon = couponMapper.selectCouponById(couponId);
        Assert.isTrue(coupon != null && Objects.equals(coupon.getState(), CouponState.ONLINE.getState()), "优惠券状态不合法");
        CouponRule couponRule = this.couponRuleResolve(custId, couponId, coupon);

        CouponUser couponUser = new CouponUser();
        couponUser.setCustId(custId);
        couponUser.setCustName(cust.getUserName());
        couponUser.setCustPhone(cust.getPhone());
        couponUser.setCouponId(couponId);
        LocalDateTime now = LocalDateTime.now();
        couponUser.setExpiredTime(this.getCouponExpiredTime(couponRule, now));
        couponUser.setReceiveTime(now);
        couponUser.setSource("赠送");
        couponUser.setType(CouponUserType.GIVE.getType());
        couponUser.setCreatedAt(now);
        couponUser.setUpdatedAt(now);
        couponUserMapper.insertCouponUser(couponUser);
    }

    private CouponRule couponRuleResolve(String custId, Integer couponId, Coupon coupon) {
        LocalDateTime now = LocalDateTime.now();
        CouponRule couponRule = couponRuleMapper.selectCouponRuleById(coupon.getRuleId());
        Assert.notNull(couponRule, "未查询到匹配的优惠券规则[id=" + coupon.getRuleId() + "]");
        if (couponRule.getUseEndTime().isBefore(now)) {
            throw new ServiceException("优惠券已失效");
        }

        // 可领取次数的校验
        if (couponRule.getUserLimit() == null || couponRule.getUserLimit() == 0) {
            return couponRule;
        }

        CouponUser couponUser = new CouponUser();
        couponUser.setCustId(custId);
        LocalDateTime startTime = null;
        if (couponRule.getDaysLimit() != null && couponRule.getDaysLimit() > 0) {
            startTime = DateUtils.addDaysAndGetBegin(now, 1 - couponRule.getDaysLimit());
        }
        Long custCouponCnt = couponUserMapper.getCustCouponCnt(custId, couponId, startTime);
        if (custCouponCnt >= couponRule.getUserLimit()) {
            String errMsg = "该优惠券只可";
            if (couponRule.getDaysLimit() != null && couponRule.getDaysLimit() > 0) {
                errMsg += "每" + couponRule.getDaysLimit() + "天";
            }
            errMsg += "领取" + couponRule.getUserLimit() + "次";
            throw new ServiceException(errMsg);
        }

        return couponRule;
    }

    private LocalDateTime getCouponExpiredTime(CouponRule couponRule, LocalDateTime now) {
        LocalDateTime expiredTime;
        if (couponRule.getRelativeTime() != null && couponRule.getRelativeTime() > 0) {
            LocalDateTime expireComputerStartTime = couponRule.getUseStartTime().isAfter(now) ? couponRule.getUseStartTime() : now;
            expiredTime = DateUtils.addDaysAndGetEnd(expireComputerStartTime, couponRule.getRelativeTime());
            expiredTime = expiredTime.isBefore(couponRule.getUseEndTime()) ? expiredTime : couponRule.getUseEndTime();
        } else {
            expiredTime = couponRule.getUseEndTime();
        }
        return expiredTime;
    }

    /**
     * 查询用户可用优惠券数量
     *
     * @param custId
     * @return
     */
    @Override
    public int getCustAvailableCouponCnt(String custId) {
        return couponUserMapper.getCustAvailableCouponCnt(custId, LocalDateTime.now(), null);
    }

    /**
     * 查询用户可用优惠券列表
     *
     * @param custId
     * @param couponUserId
     * @return
     */
    @Override
    public List<CouponVo> listCustAvailableCoupon(String custId, Integer couponUserId) {
        List<CouponVo> couponVos = couponUserMapper.listCustAvailableCoupon(custId, couponUserId, LocalDateTime.now());
        transferTabIds(couponVos);
        return couponVos;
    }

    private void transferTabIds(List<CouponVo> couponVos) {
        couponVos.forEach(couponVo -> {
            couponVo.setCategoryIds(ArrayUtil.transStrToLongList(couponVo.getCategoryIdStr()));
            couponVo.setGoodsIds(ArrayUtil.transStrToLongList(couponVo.getGoodsIdStr()));
            couponVo.setProvince(ArrayUtil.transStrToCodeList(couponVo.getProvinceStr()));
            couponVo.setCity(ArrayUtil.transStrToCodeList(couponVo.getCityStr()));
            couponVo.setArea(ArrayUtil.transStrToCodeList(couponVo.getAreaStr()));
            couponVo.setShopIds(ArrayUtil.transStrToLongList(couponVo.getShopIdStr()));
            couponVo.setWeekLimit(ArrayUtil.transStrToIntList(couponVo.getWeekLimitStr()));
        });
    }

    @Override
    public List<CouponVo> listCustCoupons(String custId, Boolean effectiveFlag) {
        List<CouponVo> couponVos = couponUserMapper.listCustCoupons(custId, effectiveFlag);
        transferTabIds(couponVos);
        return couponVos;
    }

    /**
     * 查询用户领取优惠券列表
     *
     * @param couponUser
     * @return
     */
    @Override
    public List<CouponUserVo> selectCouponUserList(CouponUser couponUser) {
        return couponUserMapper.selectCouponUserList(couponUser);
    }

    @Override
    public void resovleCouponFitable(List<CouponVo> couponVos, Order order) {
        /** 可用门店和时段的校验 */
        this.checkShopAndWeeklyFitable(couponVos, order.getShopId());
        /** 可用饮品范围的判断 */
        checkGoodsFitable(couponVos, order.getOrderDetails());
    }

    private void checkGoodsFitable(List<CouponVo> couponVos, List<OrderDetail> orderDetails) {
        List<GoodsSku> goodsSkus = this.orderDetailCheckAndAssign(orderDetails);
        List<Long> goodsIds = goodsSkus.stream().map(GoodsSku::getGoodsId).collect(Collectors.toList());
        List<Goods> goods = goodsMapper.selectGoodsByIds(goodsIds);
        Assert.notEmpty(goods, "未查询到匹配的商品信息[id=" + goodsIds + "]");
        // 订单总额
        BigDecimal orderTotalAmount = orderDetails.stream().map(OrderDetail::getAmountShould).reduce(BigDecimal.ZERO, BigDecimal::add);
        couponVos.stream().filter(couponVo -> StringUtils.isEmpty(couponVo.getNotFitableDesc())).forEach(couponVo -> {
            /** 可用饮品范围的判断 */
            List<Long> fitCouponGoodsIdList = goods.stream().filter(good ->
                    (ArrayUtil.isAllAvailable(couponVo.getCategoryIds()) || ArrayUtil.containsSplit(couponVo.getCategoryIds(), good.getCategory()))
                            || (ArrayUtil.isAllAvailable(couponVo.getGoodsIds()) || ArrayUtil.contains(couponVo.getGoodsIds(), good.getId())))
                    .map(Goods::getId).collect(Collectors.toList());
            if (CollectionUtils.isEmpty(fitCouponGoodsIdList)) {
                if (couponVo.getCategoryIds() != null && !ArrayUtil.isAllAvailable(couponVo.getCategoryIds())) {
                    couponVo.setNotFitableDesc("限定品类使用");
                } else {
                    couponVo.setNotFitableDesc("限定商品使用");
                }
                return;
            }
            /** 券价值的判断 */
            if (couponVo.getType().equals(CouponCategoryType.DEDUCTION.getType())) { // 抵扣
                if (couponVo.getPriceLimit() != null && couponVo.getPriceLimit().compareTo(BigDecimal.ZERO) != 0 && couponVo.getPriceLimit().compareTo(orderTotalAmount) > 0) {
                    couponVo.setNotFitableDesc("未达到满减要求");
                    return;
                }
                couponVo.setCouponAmount(couponVo.getPriceDiscount().min(orderTotalAmount));
            } else {
                List<OrderDetail> fitOrderDetails = orderDetails.stream().filter(orderDetail -> fitCouponGoodsIdList.contains(orderDetail.getGoodsId())).collect(Collectors.toList());
                BigDecimal fitGoodsAmount = fitOrderDetails.stream().map(OrderDetail::getAmountShould).reduce(BigDecimal.ZERO, BigDecimal::add);
                if (couponVo.getPriceLimit() != null && couponVo.getPriceLimit().compareTo(BigDecimal.ZERO) != 0 && couponVo.getPriceLimit().compareTo(fitGoodsAmount) > 0) {
                    if (couponVo.getType().equals(CouponCategoryType.DISCOUNT.getType())) { // 折扣
                        couponVo.setNotFitableDesc("未达到满减要求");
                    } else { // 免单
                        couponVo.setNotFitableDesc("未达到免单要求");
                    }
                    return;
                }

                this.wrapperCouponAmount(couponVo, fitOrderDetails, fitGoodsAmount);

                Map<Long, BigDecimal> couponAmountMap = couponVo.getCouponAmountMap();
                orderDetails.forEach(orderDetail -> {
                    BigDecimal couponAmount = couponAmountMap.get(orderDetail.getSkuId());
                    orderDetail.setCouponAmount(couponAmount == null ? BigDecimal.ZERO : couponAmount.negate());
                    orderDetail.setRealAmount(orderDetail.getAmountShould().add(orderDetail.getCouponAmount()));
                });
            }
        });
    }

    @Override
    public List<GoodsSku> orderDetailCheckAndAssign(List<OrderDetail> orderDetails) {
        Assert.notEmpty(orderDetails, "商品明细还未传递");
        List<Long> skuIds = orderDetails.stream().map(OrderDetail::getSkuId).collect(Collectors.toList());
        Assert.notEmpty(skuIds, "skuId还未传递");
        List<GoodsSku> goodsSkus = goodsSkuMapper.selectSkuListForOrder(skuIds);
        log.info("sku id size is {} and query size is {}", skuIds.size(), goodsSkus.size());
        Assert.isTrue(goodsSkus.size() == skuIds.size(), "未查询到匹配的sku记录");
        // sku的ID和价格字典
        Map<Long, GoodsSku> skuMap = goodsSkus.stream().collect(Collectors.toMap(GoodsSku::getId, Function.identity()));
        // 订单明细设置
        orderDetails.forEach(orderDetail -> {
            GoodsSku sku = skuMap.get(orderDetail.getSkuId());
            if (!Objects.equals(sku.getIsDeleted(), SkuDeleteState.NORMAL.getState())) {
                throw new ServiceException("规格已下架");
            }
            BigDecimal skuNum = new BigDecimal(orderDetail.getNum());
            orderDetail.setOriAmount(skuNum.multiply(sku.getPrice()));
            orderDetail.setUnitPrice(sku.getDiscount());
            orderDetail.setAmountShould(skuNum.multiply(sku.getDiscount()));
        });
        return goodsSkus;
    }

    private void wrapperCouponAmount(CouponVo couponVo, List<OrderDetail> fitOrderDetails, BigDecimal fitGoodsAmount) {
        Map<Long, BigDecimal> couponAmountMap = new HashMap<>();
        if (couponVo.getOrderLimit() != null && couponVo.getOrderLimit()) { // 单杯
            OrderDetail orderDetail = fitOrderDetails.stream().max(Comparator.comparing(OrderDetail::getUnitPrice)).get();
            BigDecimal discountAmount = orderDetail.getUnitPrice();
            if (couponVo.getType().equals(CouponCategoryType.DISCOUNT.getType())) { // 折扣
                discountAmount = this.getDiscountAmount(discountAmount, couponVo.getPriceDiscount()).min(BigDecimal.valueOf(50));
            }
            couponVo.setCouponAmount(discountAmount);
            couponAmountMap.put(orderDetail.getSkuId(), couponVo.getCouponAmount());
        } else { // 整单
            if (couponVo.getType().equals(CouponCategoryType.DISCOUNT.getType())) { // 折扣
                BigDecimal couponAmount = this.getDiscountAmount(fitGoodsAmount, couponVo.getPriceDiscount());
                couponAmount = couponAmount.min(BigDecimal.valueOf(50)); // 折扣类整单最高抵扣50元
                couponVo.setCouponAmount(couponAmount);
                fitOrderDetails.forEach(orderDetail -> orderDetail.setCouponAmount(this.getDiscountAmount(orderDetail.getAmountShould(), couponVo.getPriceDiscount())));
                // 订单明细优惠总和
                BigDecimal detailCouponSum = fitOrderDetails.stream().map(OrderDetail::getCouponAmount).reduce(BigDecimal.ZERO, BigDecimal::add);
                // 如果订单明细优惠总额大于订单总优惠金额，则将订单明细最后一条记录的优惠金额去掉差额
                if (detailCouponSum.compareTo(couponVo.getCouponAmount()) != 0) {
                    OrderDetail orderDetail = fitOrderDetails.get(fitOrderDetails.size() - 1);
                    orderDetail.setCouponAmount(orderDetail.getCouponAmount().add(couponVo.getCouponAmount().subtract(detailCouponSum)));
                }
            } else { // 免单
                couponVo.setCouponAmount(fitGoodsAmount);
            }
            fitOrderDetails.forEach(orderDetail -> couponAmountMap.put(orderDetail.getSkuId(), orderDetail.getCouponAmount()));
        }
        couponVo.setCouponAmountMap(couponAmountMap);
    }

    private void checkShopAndWeeklyFitable(List<CouponVo> couponVos, Long shopId) {
        Assert.notNull(shopId, "商店id还未传递");
        Shop shop = shopMapper.selectShopById(shopId);
        Assert.notNull(shop, "未查询到匹配的商店[id=" + shopId + "]");
        // 当天周几
        int dayOfWeek = Calendar.getInstance().get(Calendar.DAY_OF_WEEK);
        couponVos.forEach(couponVo -> {
            /** 可用门店的判断 */
            if (!ArrayUtil.isAllAvailable(couponVo.getProvince()) && ArrayUtil.hasContents(couponVo.getProvince())
                    && !ArrayUtil.contains(couponVo.getProvince(), shop.getProvince())) {
                couponVo.setNotFitableDesc("限定区域使用");
                return;
            }
            if (!ArrayUtil.isEmpty(couponVo.getCity()) && ArrayUtil.hasContents(couponVo.getCity()) && !ArrayUtil.contains(couponVo.getCity(), shop.getCity())) {
                couponVo.setNotFitableDesc("限定区域使用");
                return;
            }
            if (!ArrayUtil.isEmpty(couponVo.getArea()) && ArrayUtil.hasContents(couponVo.getArea()) && !ArrayUtil.contains(couponVo.getArea(), shop.getZone())) {
                couponVo.setNotFitableDesc("限定区域使用");
                return;
            }
            if (!ArrayUtil.isEmpty(couponVo.getShopIds()) && !ArrayUtil.contains(couponVo.getShopIds(), shop.getId())) {
                couponVo.setNotFitableDesc("限定门店使用");
                return;
            }
            /** 可用时段的判断 */
            if ((!ArrayUtil.isAllAvailable(couponVo.getWeekLimit()) && !ArrayUtil.contains(couponVo.getWeekLimit(), dayOfWeek)) ||
                    couponVo.getUseStartTime().isAfter(LocalDateTime.now())) {
                couponVo.setNotFitableDesc("限定时段使用");
            }
        });
    }

    private BigDecimal getDiscountAmount(BigDecimal amount, BigDecimal priceDiscount) {
        // 打折后的价格
        BigDecimal priceAfterDiscount = amount.multiply(priceDiscount.divide(BigDecimal.TEN, 4, BigDecimal.ROUND_HALF_UP)).setScale(2, BigDecimal.ROUND_HALF_UP);
        // 跟原价比取最小值
        priceAfterDiscount = priceAfterDiscount.min(amount);
        // 获取优惠金额
        return amount.subtract(priceAfterDiscount);
    }
}
