package com.soss.system.service.impl;

import com.soss.common.exception.ServiceException;
import com.soss.system.domain.Goods;
import com.soss.system.domain.ShopGoodsSku;
import com.soss.system.domain.vo.orderTaking.SkuVo;
import com.soss.system.mapper.GoodsMapper;
import com.soss.system.mapper.ShopGoodsSkuMapper;
import com.soss.system.service.IShopGoodsSkuService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.List;

/**
 * 从店铺到sku关联Service业务层处理
 *
 * @author zxq
 * @date 2022-04-28
 */
@Service
public class ShopGoodsSkuServiceImpl implements IShopGoodsSkuService {
    @Autowired
    private ShopGoodsSkuMapper shopGoodsSkuMapper;
    @Autowired
    private GoodsMapper goodsMapper;

    /**
     * 查询从店铺到sku关联
     *
     * @param id 从店铺到sku关联主键
     * @return 从店铺到sku关联
     */
    @Override
    public ShopGoodsSku selectShopGoodsSkuById(String id) {
        return shopGoodsSkuMapper.selectShopGoodsSkuById(id);
    }

    /**
     * 查询从店铺到sku关联列表
     *
     * @param shopGoodsSku 从店铺到sku关联
     * @return 从店铺到sku关联
     */
    @Override
    public List<ShopGoodsSku> selectShopGoodsSkuList(ShopGoodsSku shopGoodsSku) {
        return shopGoodsSkuMapper.selectShopGoodsSkuList(shopGoodsSku);
    }

    /**
     * 新增从店铺到sku关联
     *
     * @param shopGoodsSku 从店铺到sku关联
     * @return 结果
     */
    @Override
    public int insertShopGoodsSku(ShopGoodsSku shopGoodsSku) {
        return shopGoodsSkuMapper.insertShopGoodsSku(shopGoodsSku);
    }

    /**
     * 修改从店铺到sku关联
     *
     * @param shopGoodsSku 从店铺到sku关联
     * @return 结果
     */
    @Override
    public int updateShopGoodsSku(ShopGoodsSku shopGoodsSku) {
        return shopGoodsSkuMapper.updateShopGoodsSku(shopGoodsSku);
    }

    /**
     * 批量删除从店铺到sku关联
     *
     * @param ids 需要删除的从店铺到sku关联主键
     * @return 结果
     */
    @Override
    public int deleteShopGoodsSkuByIds(String[] ids) {
        return shopGoodsSkuMapper.deleteShopGoodsSkuByIds(ids);
    }

    /**
     * 删除从店铺到sku关联信息
     *
     * @param id 从店铺到sku关联主键
     * @return 结果
     */
    @Override
    public int deleteShopGoodsSkuById(String id) {
        return shopGoodsSkuMapper.deleteShopGoodsSkuById(id);
    }

    @Override
    public void updateShopGoodsSkuDiscount(String shopId, String goodsId, BigDecimal discount) {
        Goods goods = goodsMapper.selectGoodsById(Long.parseLong(goodsId));
        if (goods == null) {
            throw new ServiceException("商品不存在");
        }
        List<SkuVo> skus = shopGoodsSkuMapper.selectSkuByGoodAndShop(Long.parseLong(shopId), goodsId);
        if (CollectionUtils.isEmpty(skus)) {
            throw new ServiceException("sku不存在");
        }
        if (discount.compareTo(goods.getDiscount()) > 0) {
            throw new ServiceException("特惠价格不能大于原折扣价");
        }
        BigDecimal subtract = goods.getDiscount().subtract(discount); // 商品原折扣价和特惠价格差额
        if (subtract.floatValue() == 0) { // 金额没变化
            return;
        }
        skus.forEach(sku -> {
            // 重新计算店铺商品sku特惠价格
            BigDecimal newDiscount = sku.getOrigDiscount().subtract(subtract);
            shopGoodsSkuMapper.updateSkuDiscount(sku.getId(), newDiscount);
        });
    }

    @Override
    public void restoreSkuDiscount(Long shopId, Long goodsId) {
        shopGoodsSkuMapper.restoreSkuDiscount(shopId, goodsId);
    }
}
