package com.soss.web.controller.coffee;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.soss.common.annotation.Log;
import com.soss.common.core.controller.BaseController;
import com.soss.common.core.domain.AjaxResult;
import com.soss.common.core.page.TableDataInfo;
import com.soss.common.enums.BusinessType;
import com.soss.system.domain.Goods;
import com.soss.system.domain.GoodsCategory;
import com.soss.system.domain.Shop;
import com.soss.system.service.IGoodsService;
import com.soss.system.service.IShopService;
import com.soss.system.service.impl.ShopGoodsServiceImpl;
import io.swagger.annotations.ApiOperation;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.util.CollectionUtils;
import org.springframework.web.bind.annotation.*;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * 店铺Controller
 *
 * @author zxq
 * @date 2022-04-28
 */
@RestController
@RequestMapping("/system/shop")
public class ShopController extends BaseController {
    @Autowired
    private IShopService shopService;
    @Autowired
    private IGoodsService goodsService;
    @Autowired
    private ShopGoodsServiceImpl shopGoodsService;

    @GetMapping(value = "/turn")
    public AjaxResult turn(@RequestParam Long shopId, @RequestParam Long goodsId, @RequestParam String pointer) {
        return AjaxResult.success(shopService.turn(shopId, goodsId, pointer));
    }

    /**
     * 查询店铺列表
     */
    @GetMapping("/list")
    public TableDataInfo list(Shop shop) {
        startPage();
        List<Shop> list = shopService.selectShopList(shop);
        return getDataTable(list);
    }

    /**
     * 获取店铺详细信息
     */
    @PreAuthorize("@ss.hasPermi('system:shop:query')")
    @GetMapping(value = "/{id}")
    public AjaxResult getInfo(@PathVariable("id") Long id) {
        return AjaxResult.success(shopService.selectShopById(id));
    }

    /**
     * 新增店铺
     */
    @PreAuthorize("@ss.hasPermi('system:shop:add')")
    @Log(title = "店铺", businessType = BusinessType.INSERT)
    @PostMapping
    public AjaxResult add(@RequestBody Shop shop) {
        return AjaxResult.success(shopService.insertShop(shop));
    }

    /**
     * 开始测试
     */
    @PreAuthorize("@ss.hasPermi('system:shop:test:start')")
    @Log(title = "店铺", businessType = BusinessType.UPDATE)
    @GetMapping("/test/start/{shopId}")
    @ApiOperation("开始测试")
    public AjaxResult testStart(@PathVariable Long shopId) {
        shopService.testStart(shopId);
        return AjaxResult.success();
    }

    /**
     * 完成测试
     */
    @PreAuthorize("@ss.hasPermi('system:shop:test:finish')")
    @Log(title = "店铺", businessType = BusinessType.UPDATE)
    @GetMapping("/test/finish/{shopId}")
    @ApiOperation("完成测试")
    public AjaxResult testFinish(@PathVariable Long shopId) {
        shopService.testFinish(shopId);
        return AjaxResult.success();
    }

    /**
     * 开始营业
     */
    @PreAuthorize("@ss.hasPermi('system:shop:open')")
    @Log(title = "店铺", businessType = BusinessType.UPDATE)
    @GetMapping("/open/{shopId}")
    @ApiOperation("开始营业")
    public AjaxResult open(@PathVariable Long shopId) {
        shopService.open(shopId);
        return AjaxResult.success();
    }

    /**
     * 修改店铺
     */
    @PreAuthorize("@ss.hasPermi('system:shop:edit')")
    @Log(title = "店铺", businessType = BusinessType.UPDATE)
    @PutMapping
    public AjaxResult edit(@RequestBody Shop shop) {
        return toAjax(shopService.updateShop(shop));
    }

    /**
     * 删除店铺
     */
    @PreAuthorize("@ss.hasPermi('system:shop:remove')")
    @Log(title = "店铺", businessType = BusinessType.DELETE)
    @DeleteMapping("/{ids}")
    public AjaxResult remove(@PathVariable String[] ids) {
        return toAjax(shopService.deleteShopByIds(ids));
    }

    /**
     * 给店铺增加商品
     */
    @PreAuthorize("@ss.hasPermi('system:shop:addGoods')")
    @GetMapping(value = "/addGoods")
    public AjaxResult addGoods(Long shopId, String goodsIds) {
        return AjaxResult.success(shopService.addGoods(shopId, goodsIds));
    }

    @PostMapping(value = "/offGoods")
    public AjaxResult offGoods(@RequestBody String body) {
        JSONObject jsonObject = JSON.parseObject(body);
        return AjaxResult.success(shopService.offGoods(jsonObject));
    }

    /**
     * 获取店铺里面的商品信息，分类展示
     */
    @GetMapping(value = "/getCategoryGoods")
    public AjaxResult getCategoryGoods(long shopId) {
        List<GoodsCategory> categoryGoods = shopService.getCategoryGoods(shopId);
        for (GoodsCategory categoryGood : categoryGoods) {
            goodsService.resolverCategoryName(categoryGood.getGoodsList());
        }
        return AjaxResult.success(categoryGoods);
    }

    /**
     * 获取店铺里面商品对应的产品信息
     */
    @GetMapping(value = "/getSkuByGoodsId")
    public AjaxResult getSkuByGoodsId(long shopId, long goodsId) {
        return AjaxResult.success(shopService.getSkuByGoodsId(shopId, goodsId));
    }

    /**
     * 获取有店铺的城市
     *
     * @return
     */
    @GetMapping("/getShopCity")
    public AjaxResult getShopCity() {
        return AjaxResult.success(shopService.getShopCity());
    }

    @GetMapping("/area/tree")
    public AjaxResult getShopAreaTree() {
        Map<String, Map<String, Map<String, List<Shop>>>> shopAreaTree = shopService.getShopAreaTree();
        Map<String, Object> treeMap = new HashMap<>(shopAreaTree);
        return AjaxResult.success(buildShopAreaTree(treeMap));
    }

    private List<Map<String, Object>> buildShopAreaTree(Map<String, Object> dataSourceMap) {
        List<Map<String, Object>> nodeMapList = new ArrayList<>();
        dataSourceMap.forEach((key, val) -> {
            Map<String, Object> nodeMap = new HashMap<>();
            nodeMapList.add(nodeMap);
            nodeMap.put("value", key);
            if (val instanceof Map) {
                nodeMap.put("children", buildShopAreaTree((Map) val));
            } else if (val instanceof List) {
                List<Map<String, String>> shopMapList = new ArrayList<>();
                ((List) val).forEach(shop -> {
                    Map<String, String> shopMap = new HashMap<>();
                    shopMap.put("value", String.valueOf(((Shop) shop).getId()));
                    shopMap.put("label", ((Shop) shop).getName());
                    shopMapList.add(shopMap);
                });
                nodeMap.put("children", shopMapList);
            }
        });
        return nodeMapList;
    }

    @GetMapping("/goodslist")
    public TableDataInfo list(String goodsName, String category, Long shopId) {
        startPage();
        List<Goods> list = shopService.selectGoodsList(goodsName, category, shopId);
        return getDataTable(list);
    }

    @GetMapping(value = "/selectSkuByShopGoods")
    public AjaxResult selectSkuByShopGoods(long shopId, long goodsId) {
        return AjaxResult.success(shopGoodsService.selectSkuByShopGoods(shopId, goodsId));
    }

    @GetMapping("/updateShopGoodsSkuState")
    public AjaxResult updateShopGoodsSkuState(@RequestParam("ids") List<Long> shopGoodsSkuIds, @RequestParam("state") Integer state) {
        if (CollectionUtils.isEmpty(shopGoodsSkuIds)) {
            return AjaxResult.error("id列表为空");
        }
        for (Long id : shopGoodsSkuIds) {
            shopGoodsService.updateShopGoodsSkuState(id, state);
        }
        return AjaxResult.success();
    }
}
