package com.soss.system.service.impl;

import com.soss.common.enums.CouponCategoryType;
import com.soss.common.enums.CouponState;
import com.soss.common.exception.ServiceException;
import com.soss.system.domain.Coupon;
import com.soss.system.domain.CouponCategory;
import com.soss.system.mapper.CouponCategoryMapper;
import com.soss.system.mapper.CouponMapper;
import com.soss.system.service.ICouponCategoryService;
import io.jsonwebtoken.lang.Assert;
import org.apache.http.util.Asserts;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.time.LocalDateTime;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * <p>
 * 优惠券类别 服务实现类
 * </p>
 *
 * @author caiyt
 * @since 2022-07-21
 */
@Service
public class CouponCategoryServiceImpl implements ICouponCategoryService {
    @Autowired
    private CouponCategoryMapper couponCategoryMapper;
    @Autowired
    private CouponMapper couponMapper;

    /**
     * 新增优惠券类别
     *
     * @param couponCategory 优惠券类别
     * @return 结果
     */
    @Override
    @Transactional
    public void insertCouponCategory(CouponCategory couponCategory) {
        Assert.notNull(couponCategory.getName(), "优惠券类别名称还未填写");
        Assert.notNull(couponCategory.getType(), "优惠券类别的折扣形式还未选择");
        LocalDateTime now = LocalDateTime.now();
        couponCategory.setState(CouponState.ONLINE.getState());
        couponCategory.setCreatedAt(now);
        couponCategory.setUpdatedAt(now);
        couponCategoryMapper.insertCouponCategory(couponCategory);
    }

    /**
     * 修改优惠券类别
     *
     * @param couponCategory 优惠券类别
     * @return 结果
     */
    @Override
    @Transactional
    public void updateCouponCategory(CouponCategory couponCategory) {
        Asserts.notNull(couponCategory.getId(), "优惠券类别ID还未传递");
        CouponCategory couponCategoryOri = this.detail(couponCategory.getId());
        if (!couponCategoryOri.getState().equals(CouponState.OFFLINE.getState())) {
            throw new ServiceException("修改类别需先下架类别");
        }
        Assert.notNull(couponCategory.getName(), "优惠券类别名称还未填写");
        Assert.notNull(couponCategory.getType(), "优惠券类别的折扣形式还未选择");
        LocalDateTime now = LocalDateTime.now();
        couponCategory.setUpdatedAt(now);
        couponCategoryMapper.updateCouponCategory(couponCategory);
    }

    /**
     * 查询优惠券类别列表
     *
     * @param couponCategory 优惠券类别
     * @return 优惠券类别
     */
    @Override
    public List<CouponCategory> listCouponCategory(CouponCategory couponCategory) {
        List<CouponCategory> couponCategories = couponCategoryMapper.listCouponCategory(couponCategory);
        Map<Integer, Long> countMap;
        if (!CollectionUtils.isEmpty(couponCategories)) {
            List<Integer> categoryIds = couponCategories.stream().map(CouponCategory::getId).collect(Collectors.toList());
            List<CouponCategory> couponCategoryCountList = couponMapper.getCouponCategoryTypeStat(categoryIds);
            countMap = couponCategoryCountList.stream().collect(Collectors.toMap(CouponCategory::getId, CouponCategory::getCouponCnt));
        } else {
            countMap = new HashMap<>();
        }
        for (CouponCategory category : couponCategories) {
            category.setTypeDesc(CouponCategoryType.getDesc(category.getType()));
            Long count = countMap.get(category.getId());
            category.setCouponCnt(count == null ? 0L : count);
            category.setStateDesc(CouponState.getDesc(category.getState()));
        }
        return couponCategories;
    }

    /**
     * 上架优惠券类别
     *
     * @param id
     * @return
     */
    @Override
    @Transactional
    public int onlineCouponCategory(Integer id) {
        CouponCategory online = new CouponCategory(id, CouponState.ONLINE.getState());
        int updateCnt = couponCategoryMapper.updateCouponCategory(online);
        return updateCnt;
    }

    /**
     * 下架优惠券类别
     *
     * @param id
     * @return
     */
    @Override
    @Transactional
    public int offlineCouponCategory(Integer id) {
        CouponCategory online = new CouponCategory(id, CouponState.OFFLINE.getState());
        int updateCnt = couponCategoryMapper.updateCouponCategory(online);
        Coupon coupon = new Coupon();
        coupon.setCategoryId(id);
        List<Coupon> coupons = couponMapper.listCoupon(coupon);
        long unOfflineCnt = coupons.stream().filter(c -> c.getState().equals(CouponState.ONLINE.getState())).count();
        if (unOfflineCnt > 0) {
            throw new ServiceException("需类下优惠券都处于下线状态才可下线");
        }
        return updateCnt;
    }

    /**
     * 删除优惠券类别
     *
     * @param id
     * @return
     */
    @Override
    @Transactional
    public int deleteCouponCategory(Integer id) {
        CouponCategory couponCategory = this.detail(id);
        if (!couponCategory.getState().equals(CouponState.OFFLINE.getState())) {
            throw new ServiceException("需未生效状态才可删除");
        }
        CouponCategory online = new CouponCategory(id, CouponState.DELETE.getState());
        int updateCnt = couponCategoryMapper.updateCouponCategory(online);
        return updateCnt;
    }

    /**
     * 获取优惠券类别详情
     *
     * @param id
     * @return
     */
    private CouponCategory detail(Integer id) {
        CouponCategory couponCategory = couponCategoryMapper.selectCouponCategoryById(id);
        Asserts.notNull(couponCategory, "未查询到匹配记录[id=" + couponCategory + "]");
        return couponCategory;
    }
}
