package com.soss.system.service.impl;

import com.soss.system.constants.SystemConstant;
import com.soss.system.domain.CouponRule;
import com.soss.system.domain.Goods;
import com.soss.system.domain.GoodsCategory;
import com.soss.system.domain.Shop;
import com.soss.system.domain.vo.CouponVo;
import com.soss.system.mapper.CouponRuleMapper;
import com.soss.system.mapper.GoodsCategoryMapper;
import com.soss.system.mapper.GoodsMapper;
import com.soss.system.mapper.ShopMapper;
import com.soss.system.service.ICouponRuleService;
import com.soss.system.utils.AreaUtil;
import com.soss.system.utils.ArrayUtil;
import io.jsonwebtoken.lang.Assert;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * <p>
 * 优惠券规则 服务实现类
 * </p>
 *
 * @author caiyt
 * @since 2022-07-21
 */
@Service
public class CouponRuleServiceImpl implements ICouponRuleService {
    @Autowired
    private CouponRuleMapper couponRuleMapper;
    @Autowired
    private GoodsCategoryMapper goodsCategoryMapper;
    @Autowired
    private GoodsMapper goodsMapper;
    @Autowired
    private ShopMapper shopMapper;
    @Autowired
    private AreaUtil areaUtil;

    @Override
    public Map<Integer, CouponRule> getCouponRuleMap(List<Integer> ids) {
        List<CouponRule> couponRules = couponRuleMapper.selectCouponRuleByIds(ids);
        return couponRules.stream().collect(Collectors.toMap(CouponRule::getId, Function.identity()));
    }

    /**
     * 查询优惠券规则详情
     *
     * @param id
     * @return
     */
    @Override
    public CouponRule detail(Integer id) {
        CouponRule couponRule = couponRuleMapper.selectCouponRuleById(id);
        Assert.notNull(couponRule, "未查询到匹配的优惠券规则记录[id=" + id + "]");
        return couponRule;
    }

    @Override
    public void translateUseLimit(List<CouponVo> couponVos) {
        if (CollectionUtils.isEmpty(couponVos)) {
            return;
        }
        transCategoryIds(couponVos);
        translateArea(couponVos);
        translateWeekLimit(couponVos);
    }

    private void translateWeekLimit(List<CouponVo> couponVos) {
        for (CouponVo couponVo : couponVos) {
            if (!ArrayUtil.isAllAvailable(couponVo.getWeekLimit())) {
                String weekLimitDesc = Arrays.stream(couponVo.getWeekLimit())
                        .map(week -> "每" + SystemConstant.WEEK_DESC[week - 1]).collect(Collectors.joining("，"));
                couponVo.setWeekLimitDesc(weekLimitDesc);
            }
        }
    }

    private void transCategoryIds(List<CouponVo> couponVos) {
        List<Long> categoryIds = new ArrayList<>(16);
        List<Long> goodsIds = new ArrayList<>(16);
        for (CouponVo couponVo : couponVos) {
            if (!ArrayUtil.isAllAvailable(couponVo.getCategoryIds())) {
                ArrayUtil.listAddArray(categoryIds, couponVo.getCategoryIds());
            }
            ArrayUtil.listAddArray(goodsIds, couponVo.getGoodsIds());
        }

        Map<Long, String> categoryMap;
        if (!CollectionUtils.isEmpty(categoryIds)) {
            List<GoodsCategory> goodsCategories = goodsCategoryMapper.selectGoodsCategoryByIds(categoryIds);
            categoryMap = goodsCategories.stream().collect(Collectors.toMap(GoodsCategory::getId, GoodsCategory::getName));
        } else {
            categoryMap = new HashMap<>(0);
        }

        Map<Long, String> goodsMap;
        if (!CollectionUtils.isEmpty(goodsIds)) {
            List<Goods> goods = goodsMapper.selectGoodsByIds(goodsIds);
            goodsMap = goods.stream().collect(Collectors.toMap(Goods::getId, Goods::getName));
        } else {
            goodsMap = new HashMap<>(0);
        }

        for (CouponVo couponVo : couponVos) {
            if (ArrayUtil.isAllAvailable(couponVo.getCategoryIds())) {
                couponVo.setGoodLimitDesc("全品类");
                continue;
            } else if (ArrayUtil.hasContents(couponVo.getCategoryIds())) {
                String categoryDec = Arrays.stream(couponVo.getCategoryIds()).map(categoryMap::get).collect(Collectors.joining(","));
                couponVo.setGoodLimitDesc(categoryDec);
                continue;
            }

            if (ArrayUtil.hasContents(couponVo.getGoodsIds())) {
                String goodsDesc = Arrays.stream(couponVo.getGoodsIds()).map(goodsMap::get).collect(Collectors.joining(","));
                couponVo.setGoodLimitDesc(goodsDesc);
            }
        }
    }

    private void translateArea(List<CouponVo> couponVos) {
        List<String> province = new ArrayList<>(16);
        List<String> city = new ArrayList<>(16);
        List<String> area = new ArrayList<>(16);
        List<Long> shopIds = new ArrayList<>(16);
        for (CouponVo couponVo : couponVos) {
            if (!ArrayUtil.isAllAvailable(couponVo.getProvince())) {
                ArrayUtil.listAddArray(province, couponVo.getProvince());
            }
            ArrayUtil.listAddArray(city, couponVo.getCity());
            ArrayUtil.listAddArray(area, couponVo.getArea());
            ArrayUtil.listAddArray(shopIds, couponVo.getShopIds());
        }

        Map<Long, Shop> shopMapInit;
        if (!CollectionUtils.isEmpty(shopIds)) {
            List<Shop> shops = shopMapper.selectShopByIds(shopIds);
            shopMapInit = shops.stream().collect(Collectors.toMap(Shop::getId, Function.identity()));
        } else {
            shopMapInit = new HashMap<>(0);
        }
        Map<Long, Shop> shopMap = shopMapInit;

        for (CouponVo couponVo : couponVos) {
            if (ArrayUtil.isAllAvailable(couponVo.getProvince())) {
                couponVo.setAreaLimitDesc("全国通用");
                continue;
            } else if (ArrayUtil.hasContents(couponVo.getProvince())) {
                String provDesc = Arrays.stream(couponVo.getProvince()).map(areaUtil::getProvNameByCode).collect(Collectors.joining(","));
                couponVo.setAreaLimitDesc(provDesc);
                continue;
            }

            if (ArrayUtil.hasContents(couponVo.getCity())) {
                String cityDesc = Arrays.stream(couponVo.getCity()).map(areaUtil::getCityOrAreaNameByCode).collect(Collectors.joining(","));
                couponVo.setAreaLimitDesc(cityDesc);
                continue;
            }

            if (ArrayUtil.hasContents(couponVo.getArea())) {
                String areaDesc = Arrays.stream(couponVo.getArea()).map(areaUtil::getCityOrAreaNameByCode).collect(Collectors.joining(","));
                couponVo.setAreaLimitDesc(areaDesc);
                continue;
            }

            if (!ArrayUtil.hasContents(couponVo.getShopIds())) {
                continue;
            }

            String shopDesc = Arrays.stream(couponVo.getShopIds()).map(shopId -> {
                Shop shop = shopMap.get(shopId);
                if (shop == null) {
                    return null;
                }
                return areaUtil.getCityOrAreaNameByCode(shop.getZone()) + shop.getName();
            }).filter(Objects::nonNull).collect(Collectors.joining(","));
            couponVo.setAreaLimitDesc(shopDesc);
        }
    }
}
