package com.soss.system.service.impl;

import java.util.Arrays;
import java.util.Date;
import java.util.List;

import com.alibaba.fastjson.JSONObject;
import com.soss.common.core.domain.AjaxResult;
import com.soss.common.exception.ServiceException;
import com.soss.common.utils.GenerateCode;
import com.soss.common.utils.StringUtils;
import com.soss.system.constants.OrderStatusConstant;
import com.soss.system.domain.Order;
import com.soss.system.domain.OrderDetail;
import com.soss.system.domain.OrderSnapshot;
import com.soss.system.domain.vo.OrderQuery;
import com.soss.system.jiguang.impl.JiGuangPushServiceImpl;
import com.soss.system.mapper.OrderMapper;
import com.soss.system.mapper.OrderSnapshotMapper;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import com.soss.system.mapper.OrderRefundMapper;
import com.soss.system.domain.OrderRefund;
import com.soss.system.service.IOrderRefundService;

/**
 * 订单退款Service业务层处理
 * 
 * @author zxq
 * @date 2022-04-28
 */
@Service
public class OrderRefundServiceImpl implements IOrderRefundService 
{
    @Autowired
    private OrderRefundMapper orderRefundMapper;
    @Autowired
    private OrderServiceImpl orderService;

    @Autowired
    private OrderSnapshotMapper orderSnapshotMapper;
    @Autowired
    private MachineApiServiceImpl machineApiService;
    @Autowired
    private JiGuangPushServiceImpl jiGuangPushService;
    @Autowired
    private OrderOperationLogServiceImpl orderOperationLogService;

    /**
     * 查询订单退款
     * 
     * @param id 订单退款主键
     * @return 订单退款
     */
    @Override
    public OrderRefund selectOrderRefundById(String id)
    {
        OrderRefund orderRefund = orderRefundMapper.selectOrderRefundById(id);
        Order order = orderService.selectOrderById(orderRefund.getOrderId());
        OrderSnapshot orderSnapshot = orderSnapshotMapper.selectOrderSnapshotByOrderId(order.getId());
        List<OrderDetail> orderDetails = JSONObject.parseArray(orderSnapshot.getSnapshot(), OrderDetail.class);
        order.setOrderDetails(orderDetails);
        orderRefund.setOrder(order);
        return orderRefund;
    }

    /**
     * 查询订单退款列表
     * 
     * @param orderRefund 订单退款
     * @return 订单退款
     */
    @Override
    public List<OrderRefund> selectOrderRefundList(OrderRefund orderRefund)
    {
        List<OrderRefund> orderRefunds = orderRefundMapper.selectOrderRefundList(orderRefund);
        for (OrderRefund refund : orderRefunds) {
            Order order = orderService.selectOrderById(refund.getOrderId());
            OrderSnapshot orderSnapshot = orderSnapshotMapper.selectOrderSnapshotByOrderId(order.getId());
            List<OrderDetail> orderDetails = JSONObject.parseArray(orderSnapshot.getSnapshot(), OrderDetail.class);
            order.setOrderDetails(orderDetails);
            refund.setOrder(order);
        }
        return orderRefunds;

    }

    /**
     * 新增订单退款
     * 
     * @param orderRefund 订单退款
     * @return 结果
     */
    @Override
    public AjaxResult insertOrderRefund(OrderRefund orderRefund)
    {
        Order order = orderService.selectById(orderRefund.getOrderId());
        if(order ==null ){
          return AjaxResult.error("找不到该订单");
        }
       // 制作中、制作完成、聚餐中、已完成
        String state = order.getState();
        List<String> status = Arrays.asList(OrderStatusConstant.production,OrderStatusConstant.productionCompleted,OrderStatusConstant.Taking,OrderStatusConstant.completed);
        if(OrderStatusConstant.Paid.equals(state)){
            orderRefund.setRefundAmount(order.getAmount());
           // 4 支付制作中 5 制作完成未取 6 取餐中 7 取餐完成
        }else if(status.contains(state)){
            String desc = orderRefund.getDesc();
            if(StringUtils.isEmpty(desc)){
                return AjaxResult.error("请联系客服进行退款");
            }
        }else{
            return AjaxResult.error("该订单状态不允许退款");
        }

        orderRefund.setCreatedAt(new Date());
        orderRefund.setUpdatedAt(new Date());
        orderRefund.setState("0");
        orderRefund.setOrderNo(order.getOrderNo());
        orderRefund.setRefundNo(GenerateCode.getCode("R","%09d"));
        orderRefund.setTotalFee(order.getAmount());
        orderRefund.setOrderNo(order.getOrderNo());
        if(StringUtils.isEmpty(orderRefund.getCreateUserName())){
            orderRefund.setCreateUserName(order.getUserName());
        }
        int i = orderRefundMapper.insertOrderRefund(orderRefund);
        order.setState(OrderStatusConstant.refunding);
        orderService.updateOrder(order);
        orderRefund.setOrder(order);
        return AjaxResult.success(orderRefund);
    }
    public void cancel(Order order) {
        if(OrderStatusConstant.Unpaid.equals(order.getState())){
            order.setState(OrderStatusConstant.UnpaidCancel);
            order.setUpdatedAt(new Date());
            orderService.updateOrder(order);
        }else if(OrderStatusConstant.Paid.equals(order.getState())){
            order.setState(OrderStatusConstant.PaidCancel);
            order.setUpdatedAt(new Date());
            orderService.updateOrder(order);
        }else{
            throw new ServiceException("该订单不允许退款");
        }
    }
    /**
     * 修改订单退款
     * 
     * @param orderRefund 订单退款
     * @return 结果
     */
    @Override
    public int updateOrderRefund(OrderRefund orderRefund)
    {
        return orderRefundMapper.updateOrderRefund(orderRefund);
    }

    /**
     * 批量删除订单退款
     * 
     * @param ids 需要删除的订单退款主键
     * @return 结果
     */
    @Override
    public int deleteOrderRefundByIds(String[] ids)
    {
        return orderRefundMapper.deleteOrderRefundByIds(ids);
    }

    /**
     * 删除订单退款信息
     * 
     * @param id 订单退款主键
     * @return 结果
     */
    @Override
    public int deleteOrderRefundById(String id)
    {
        return orderRefundMapper.deleteOrderRefundById(id);
    }

    @Override
    public OrderRefund getOrderRefund(String id) {
        OrderRefund orderRefund = orderRefundMapper.selectOrderRefundById(id);
        if("0".equals(orderRefund.getState()) ||"3".equals(orderRefund.getState())){

        }else{
            throw new ServiceException("该状态不允许退款");
        }
        Order order = orderService.selectById(orderRefund.getOrderId());
        orderRefund.setTotalFee(order.getAmount());
        return orderRefund;
    }

    @Override
    public List<OrderRefund> selectOrderRefundList(OrderQuery orderQuery) {
        List<OrderRefund> orderRefunds = orderRefundMapper.selectList(orderQuery);
        for (OrderRefund refund : orderRefunds) {
            Order order = orderService.selectOrderById(refund.getOrderId());
            OrderSnapshot orderSnapshot = orderSnapshotMapper.selectOrderSnapshotByOrderId(order.getId());
            List<OrderDetail> orderDetails = JSONObject.parseArray(orderSnapshot.getSnapshot(), OrderDetail.class);
            order.setOrderDetails(orderDetails);
            refund.setOrder(order);
        }
        return orderRefunds;
    }
}
