package com.soss.system.service.impl;

import java.math.BigDecimal;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.soss.common.constant.HttpStatus;
import com.soss.common.core.domain.AjaxResult;
import com.soss.common.core.domain.model.LoginUser;
import com.soss.common.exception.ServiceException;
import com.soss.common.utils.StringUtils;
import com.soss.common.utils.spring.SpringUtils;
import com.soss.system.constants.OrderStatusConstant;
import com.soss.system.domain.*;
import com.soss.system.domain.vo.orderTaking.CategoryVo;
import com.soss.system.domain.vo.orderTaking.GoodsVo;
import com.soss.system.domain.vo.orderTaking.OrderTakingVo;
import com.soss.system.domain.vo.orderTaking.SkuVo;
import com.soss.system.jiguang.impl.JiGuangPushServiceImpl;
import com.soss.system.service.IOrderService;
import com.soss.system.domain.vo.OrderQuery;
import com.soss.system.mapper.*;
import com.soss.system.weixin.util.SendMessageUtils;
import org.aspectj.weaver.ast.Or;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.data.redis.support.atomic.RedisAtomicInteger;
import org.springframework.security.core.token.TokenService;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

/**
 * 订单Service业务层处理
 * 
 * @author zxq
 * @date 2022-04-28
 */
@Service
public class OrderServiceImpl implements IOrderService
{
    @Autowired
    private OrderMapper orderMapper;
    @Autowired
    private ShopGoodsSkuMapper shopGoodsSkuMapper;
    @Autowired
    private OrderDetailMapper orderDetailMapper;
    @Autowired
    private GoodsMapper goodsMapper;
    @Autowired
    private MachineMapper machineMapper;
    @Autowired
    private MachineApiServiceImpl machineApiService;
    @Autowired
    private ShopMapper shopMapper;
    @Autowired
    private OrderOperationLogServiceImpl orderOperationLogService;
    @Autowired
    private OrderSnapshotServiceImpl orderSnapshotService;
    @Autowired
    private JiGuangPushServiceImpl jiGuangPushService;
    @Autowired
    private WechatMessageServiceImpl wechatMessageService;
    @Autowired
    private SendMessageUtils sendMessageUtils;
    @Autowired
    private OrderTakingServiceImpl orderTakingService;



    /**
     * 查询订单
     * 
     * @param id 订单主键
     * @return 订单
     */
    @Override
    public Order selectOrderById(String id)
    {
        Order order = orderMapper.selectOrderById(id);
        order.setShop(shopMapper.selectShopById(order.getShopId()));
        OrderSnapshot orderSnapshot = orderSnapshotService.selectOrderSnapshotByOrderId(order.getId());
        String snapshot = orderSnapshot.getSnapshot();
        List<OrderDetail> orderDetails1 = JSONObject.parseArray(snapshot, OrderDetail.class);
        order.setOrderDetails(orderDetails1);
        return order;
    }

    public Order selectById(String id){
        return orderMapper.selectOrderById(id);
    }

    /**
     * 查询订单列表
     * 
     * @param order 订单
     * @return 订单
     */
    @Override
    public List<Order> selectOrderList(OrderQuery order)
    {
        List<Order> orders = orderMapper.selectOrderByQuery(order);
        for (Order order1 : orders) {
            order1.setShop(shopMapper.selectShopById(order1.getShopId()));
        }
        return orders;
    }
    public List<Order> selectOrderList(Order order)
    {
        List<Order> orders = orderMapper.selectOrderList(order);

        return orders;
    }

    /**
     * 新增订单
     * 
     *
     * @param loginUser
     * @param order 订单
     * @return 结果
     */
    @Override
    @Transactional
    public AjaxResult insertOrder(LoginUser loginUser, Order order)
    {
        AjaxResult result = check(order);
        if(HttpStatus.SUCCESS != (int)result.get(result.CODE_TAG)){
            return result;
        }
        order.setState(OrderStatusConstant.Unpaid);
        order.setUserId(loginUser.getOpenId());
        order.setUserName(loginUser.getUsername());
        order.setCreatedAt(new Date());
        order.setOrderNum(getOrderNum(order.getShopId()));
        order.setOrderNo(getOrderNo(order.getSource()));
        order.setUpdatedAt(new Date());
        String machineId = null;
        if(StringUtils.isEmpty(order.getMachineId())){
            Machine machine =new Machine();
            machine.setShopId(Long.parseLong(order.getShopId()));
            List<Machine> machines = machineMapper.selectMachineList(machine);
            if(machines!=null &&!machines.isEmpty()){
                Machine machine1 = machines.get(0);
                if(!"1".equals(machine1.getState())){
                    return AjaxResult.error("机器暂时不可用");
                }
                machineId = machine1.getId();
                order.setMachineId(machineId);
            }else{
                return AjaxResult.error("该店铺暂无绑定机器");
            }
        }

        List<OrderDetail> orderDetails = order.getOrderDetails();
        if(orderDetails !=null && !orderDetails.isEmpty()){
//            BigDecimal bigDecimal = new BigDecimal(0);
//            for (OrderDetail orderDetail : orderDetails) {
//                bigDecimal =  bigDecimal.add(orderDetail.getRealAmount().multiply( new BigDecimal(orderDetail.getNum())));
//            }
//            if(order.getAmount().compareTo(bigDecimal)!=0){
//                throw new ServiceException("价格计算出现了问题，请重新计算"+ bigDecimal.stripTrailingZeros().toString() );
//            }
//            order.setAmount(bigDecimal);
            int  i = orderMapper.insertOrder(order);
            for (OrderDetail orderDetail : orderDetails) {
                orderDetail.setOrderId(order.getId());
                orderDetail.setCreatedAt(new Date());
                orderDetail.setUpdatedAt(new Date());
                orderDetail.setMachineId(machineId);
                orderDetailMapper.insertOrderDetail(orderDetail);
            }
        }
        //发送数据给机器

        machineApiService.createOrder(order);
        orderOperationLogService.insertOrderOperationLog(order);
        orderSnapshotService.saveOrderSnapshot(order);
        return AjaxResult.success(order.getOrderNo());

    }

    /**
     *订单重写从这里开始，这里要承担大部分的业务逻辑，逻辑集中化处理
     */
    @Override
    public int updateOrder(Order order)
    {
        String state = order.getState();
        switch (state){
            case OrderStatusConstant.Paid:
                paidState(order);
                break;
            case OrderStatusConstant.production:
                productionState(order);
                break;
            case OrderStatusConstant.productionCompleted:
                productionCompletedState(order);
                break;
            case OrderStatusConstant.Taking:
                takingState(order);
                break;
            case OrderStatusConstant.completed:
                completedState(order);
                break;
            case OrderStatusConstant.UnpaidCancel:
            case OrderStatusConstant.PaidCancel:
                unpaidCancelState(order);
                break;
            case OrderStatusConstant.refunding:
                refundingState(order);
                break;
            case OrderStatusConstant.PartialRefund:
            case OrderStatusConstant.refund:
                partialRefundState(order);
                break;
        }

        order.setUpdatedAt(new Date());
        return orderMapper.updateOrder(order);
    }

    /**
     * 部分退款
     * @param order
     */
    private void partialRefundState(Order order) {
        jiGuangPushService.pushOrderState(order);
        machineApiService.updateOrder(order);
    }

    /**
     * 制作完成 机器给我们
     * 记录日志
     * 极光推送
     * @param order
     */
    private void completedState(Order order) {
        order.setFinishTime(new Date());
        orderOperationLogService.insertOrderOperationLog(order);
        jiGuangPushService.pushOrderState(order);
    }

    /**
     * 退款中
     * 极光推送
     * 推送给机器
     * @param order
     */
    private void refundingState(Order order) {
        jiGuangPushService.pushOrderState(order);
        machineApiService.updateOrder(order);
    }

    /**
     * 未支付取消  支付取消
     * 记录日志
     * 发送给机器
     * 极光推送
     * @param order
     */
    private void unpaidCancelState(Order order) {
        orderOperationLogService.insertOrderOperationLog(order);
        machineApiService.updateOrder(order);
        jiGuangPushService.pushOrderState(order);

    }

    /**
     * 取餐中
     * 极光推送
     * @param order
     */
    private void takingState(Order order) {
        orderOperationLogService.insertOrderOperationLog(order);
        jiGuangPushService.pushOrderState(order);
    }

    /**
     * 制作完成待取餐
     * 记录日志
     * 极光推送
     * 消息记录
     * 发送推送消息
     * @param order
     */
    private void productionCompletedState(Order order) {
        orderOperationLogService.insertOrderOperationLog(order);
        jiGuangPushService.pushOrderState(order);
        wechatMessageService.insertWechatMessage(order.getUserId(),"制作完成,您的订单已经制作完成，现在可以去取餐啦！");
        sendMessageUtils.sendWxMsg(order);

    }

    /**
     * 制作中，机器推送给我们
     * 记录日志
     * 发送消息给安卓
     * 消息记录
     * 发送推送消息
     * @param order
     */
    private void productionState(Order order) {
        orderOperationLogService.insertOrderOperationLog(order);
        jiGuangPushService.pushOrderState(order);
        wechatMessageService.insertWechatMessage(order.getUserId(),"制作提醒,您的订单已经开始制作，马上就能享受美味了！");
        sendMessageUtils.sendWxMsg(order);

    }

    /**
     * 已付款要干这四件事
     * 记录日志，
     * 发送给机器
     * 发送给安卓
     * 消息记录
     * 发送推送消息
     * @param order
     */
    private void paidState(Order order) {
        orderOperationLogService.insertOrderOperationLog(order);
        machineApiService.updateOrder(order);
        //极光推送
        jiGuangPushService.pushOrderState(order);
        wechatMessageService.insertWechatMessage(order.getUserId(),"下单成功,您已经下单成功了~");
        sendMessageUtils.sendWxMsg(order);
    }

    /**
     * 批量删除订单
     * 
     * @param ids 需要删除的订单主键
     * @return 结果
     */
    @Override
    public int deleteOrderByIds(String[] ids)
    {
        return orderMapper.deleteOrderByIds(ids);
    }

    /**
     * 删除订单信息
     * 
     * @param id 订单主键
     * @return 结果
     */
    @Override
    public int deleteOrderById(String id)
    {
        return orderMapper.deleteOrderById(id);
    }

    @Override
    public AjaxResult check(Order order) {
        //验证订单状态
        List<OrderDetail> orderDetails = order.getOrderDetails();
        if(orderDetails==null ||orderDetails.isEmpty()){
            return AjaxResult.error("订单信息不能为空");
        }

        for (OrderDetail orderDetail : orderDetails) {
            ShopGoodsSku shopGoodsSku = new ShopGoodsSku();
            shopGoodsSku.setSkuId(Long.parseLong(orderDetail.getSkuId()));
            shopGoodsSku.setShopId(Long.valueOf(order.getShopId()));
            shopGoodsSku.setIsDeleted("0");
            List<ShopGoodsSku> shopGoodsSkus = shopGoodsSkuMapper.selectShopGoodsSkuList(shopGoodsSku);
            if(shopGoodsSkus ==null ||shopGoodsSkus.isEmpty()){
                return AjaxResult.error("该商品已不存在："+orderDetail.getGoodsName());
            }
            ShopGoodsSku shopGoodsSku1 = shopGoodsSkus.get(0);
            if(!"1".equals(shopGoodsSku1.getState())){
                return AjaxResult.error("该商品所选种类已经告罄，请重新进行选择："+orderDetail.getGoodsName());
            }
        }
        return AjaxResult.success();
    }

    @Override
    public List<Order> getMyOrder(String openId) {

        List<Order> orders = orderMapper.selectByUserId(openId);
        for (Order order : orders) {
            order.setShop(shopMapper.selectShopById(order.getShopId()));
            OrderSnapshot orderSnapshot = orderSnapshotService.selectOrderSnapshotByOrderId(order.getId());
            order.setOrderDetails(JSONArray.parseArray(orderSnapshot.getSnapshot(),OrderDetail.class));
        }
        return orders;
    }

    @Override
    @Transactional
    public int cancel(String orderId) {
        Order order = orderMapper.selectOrderById(orderId);
        if(OrderStatusConstant.Unpaid.equals(order.getState())){
            order.setState("8");
            order.setUpdatedAt(new Date());
            this.updateOrder(order);

        }else if(OrderStatusConstant.Paid.equals(order.getState())){
            order.setState("9");
            order.setUpdatedAt(new Date());
            this.updateOrder(order);
        }else{
            throw new ServiceException("该订单不允许取消");
        }

        return 0;
    }



    /**
     * 获取机器顺序号
     * @param shopId
     * @return
     */
    private String getOrderNum(String shopId){
        Calendar calendar = Calendar.getInstance();
        String key = calendar.getTime().toString().substring(0,3);
        SimpleDateFormat simpleDateFormat = new SimpleDateFormat("yyyy-MM-dd");
        String format = simpleDateFormat.format(new Date());
        String redisKey = "shopId"+format+shopId;
        RedisAtomicInteger redisAtomicInteger = new RedisAtomicInteger(redisKey,SpringUtils.getBean(StringRedisTemplate.class).getConnectionFactory());
        if(0 == redisAtomicInteger.get()){
            redisAtomicInteger.set(30);
        }
        redisAtomicInteger.expire(1, TimeUnit.DAYS);
        int andIncrement = redisAtomicInteger.getAndIncrement();
        return key+String.format("%03d",andIncrement);
    }
    //获取订单号
    public String getOrderNo(Integer source){
        String channel = "M";
        if(source !=null){
            if(source == 1){
                channel="A";
            }else if(source ==2){
                channel="B";
            }
        }
        SimpleDateFormat simpleDateFormat = new SimpleDateFormat("yyyyMMddHH");
        String format = simpleDateFormat.format(new Date());
        format = format.substring(2);
        String key = "orderNo";
        RedisAtomicInteger redisAtomicInteger = new RedisAtomicInteger(key,SpringUtils.getBean(StringRedisTemplate.class).getConnectionFactory());
        int andIncrement = redisAtomicInteger.getAndIncrement();
        if(andIncrement>9999){
            redisAtomicInteger.set(0);
            andIncrement = 0;
        }
        String format1 = String.format("%04d", andIncrement);
        int randomNumber = new Random().nextInt(99);
        return "N"+format+channel+format1+randomNumber;
    }

    public  List<JSONObject>  getOrderInfo(String machineCode) {
        Machine machine = machineMapper.selectMachineByCode(machineCode);
        List<String> status = Arrays.asList("3","4","5");
        List<Order> orders = orderMapper.selectOrder(status, machine.getId());
        Map<String, List<Order>> collect = orders.stream().collect(Collectors.groupingBy(Order::getState));
        List<JSONObject> objects = new ArrayList<>();
        for(String key:collect.keySet()){
            JSONObject jsonObject =new JSONObject();
            objects.add(jsonObject);
            jsonObject.put("state",key);
            List<Order> orders1 = collect.get(key);
            List<JSONObject> jsonObjects = new ArrayList<>();
            jsonObject.put("orders",jsonObjects);
            for (Order order : orders1) {
                JSONObject jsonObject1 = new JSONObject();
                jsonObjects.add(jsonObject1);
                jsonObject1.put("orderId",order.getId());
                jsonObject1.put("orderNum",order.getOrderNum());
                jsonObject1.put("orderNo",order.getOrderNo());
            }
        }
        return objects;

    }


    public String refundOrder(String orderId) {
        Order order = orderMapper.selectOrderById(orderId);
        List<String> status = Arrays.asList("2","3");
        if(status.contains(order.getState())){
            //todo 创建退款单，进行退款
        }else{
            //todo 创建退款单

        }
        return null;
    }

    public Map<String,String> getMyFristOrder(String openId) {
        List<String> status = Arrays.asList("2","3","4","7");
        Order order = orderMapper.selectHomeByUserId(openId,status);
        if(order ==null){
            return null;
        }
        OrderDetail orderDetail = new OrderDetail();
        orderDetail.setOrderId(order.getId());
        List<OrderDetail> orderDetails = orderDetailMapper.selectOrderDetailList(orderDetail);
        order.setOrderDetails(orderDetails);
        String waitTime = getWaitTime(order);
        Map<String,String> map =new HashMap<>();
        map.put("waitTime",waitTime);
        map.put("pickCode",order.getPickCode());
        map.put("orderNum",order.getOrderNum());
        map.put("state",order.getState());
        return map;

    }

    public String getWaitTime(Order order) {
        String shopId = order.getShopId();
        List<String> status = Arrays.asList(OrderStatusConstant.Paid,OrderStatusConstant.production);
        List<Order> orders = orderMapper.selectOrderByShopId(status, shopId);
        List<OrderDetail> list = new ArrayList<>();
        list.addAll(order.getOrderDetails());
        for (Order order1 : orders) {
            OrderDetail orderDetail = new OrderDetail();
            orderDetail.setOrderId(order1.getId());
            List<OrderDetail> orderDetails = orderDetailMapper.selectOrderDetailList(orderDetail);
            if(CollectionUtils.isEmpty(orderDetails)){

            }else{
                list.addAll(orderDetails);
            }
        }
        Long takeTimeCount =0L;
        for (OrderDetail orderDetail : list) {
            Long takeTime = goodsMapper.selectGoodsById(orderDetail.getGoodsId()).getTakeTime();
            takeTime = Integer.parseInt(orderDetail.getNum())*takeTime;
            takeTimeCount+=takeTime;
        }

        return  String.valueOf((int)(takeTimeCount/60));

    }

    public  List<GoodsVo> getNextOrder(String orderId,String shopId) {
        OrderDetail orderDetail = new OrderDetail();
        orderDetail.setOrderId(Long.parseLong(orderId));
        List<OrderDetail> orderDetails = orderDetailMapper.selectOrderDetailList(orderDetail);
        List<String> skuIds = orderDetails.stream().map(OrderDetail::getSkuId).collect(Collectors.toList());
        OrderTakingVo infoByShop = orderTakingService.getInfoByShop(shopId);
        List<CategoryVo> categorys = infoByShop.getCategorys();
        List<GoodsVo> ccs = new ArrayList<>();
        if(categorys !=null && !categorys.isEmpty()){
            for (CategoryVo category : categorys) {
                if(category.getId()==0 ){
                    continue;
                }
                List<GoodsVo> goods = category.getGoods();
                if(goods==null ||goods.isEmpty()){
                    continue;
                }
                for (GoodsVo good : goods) {
                    List<SkuVo> skuVoList = new ArrayList<>();
                    if(good.getSkus()==null ||good.getSkus().isEmpty()){
                        continue;
                    }
                    for (SkuVo skus : good.getSkus()) {
                        if(skuIds.contains(skus.getSkuId())){
                            skuVoList.add(skus);
                        }
                    }
                    if(!skuVoList.isEmpty()){
                        ccs.add(good);
                        good.setSkus(skuVoList);
                    }
                }
                
            }
        }
        return ccs;



    }
}
