package com.soss.web.controller.coffee;

import com.soss.common.annotation.Log;
import com.soss.common.core.controller.BaseController;
import com.soss.common.core.domain.AjaxResult;
import com.soss.common.core.page.TableDataInfo;
import com.soss.common.enums.BusinessType;
import com.soss.common.enums.SkuDeleteState;
import com.soss.common.exception.ServiceException;
import com.soss.system.domain.Goods;
import com.soss.system.domain.GoodsSku;
import com.soss.system.service.IGoodsService;
import com.soss.system.service.IGoodsSkuService;
import io.jsonwebtoken.lang.Assert;
import io.swagger.annotations.ApiOperation;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.web.bind.annotation.*;

import java.util.List;
import java.util.Objects;

/**
 * 商品Controller
 *
 * @author zxq
 * @date 2022-04-28
 */
@RestController
@RequestMapping("/system/goods")
public class GoodsController extends BaseController {
    @Autowired
    private IGoodsService goodsService;
    @Autowired
    private IGoodsSkuService goodsSkuService;

    /**
     * 查询商品列表
     */
    @PreAuthorize("@ss.hasPermi('system:goods:list')")
    @GetMapping("/list")
    public TableDataInfo list(Goods goods) {
        startPage();
        List<Goods> list = goodsService.selectGoodsList(goods);
        return getDataTable(list);
    }

    /**
     * 是否全部上架
     * @param goodsId
     * @param boo
     * @return
     */
    @GetMapping("/putShop")
    public AjaxResult  putShop(@RequestParam("goodsId") Long goodsId, @RequestParam("boo") boolean boo){
        try{
            return AjaxResult.success(goodsService.putShop(goodsId,boo));
        }catch (ServiceException serviceException){
            return AjaxResult.error(99,null);
        }

    }

    /**
     * 下架
     * @param goodsId
     * @param boo
     * @return
     */
    @GetMapping("/offShop")
    public AjaxResult offShop(@RequestParam("goodsId") Long goodsId,@RequestParam(required = false) boolean boo){
        return AjaxResult.success(goodsService.offShop(goodsId,boo));
    }

    /**
     * 获取商品详细信息
     */
    @PreAuthorize("@ss.hasPermi('system:goods:query')")
    @GetMapping(value = "/{id}")
    public AjaxResult getInfo(@PathVariable("id") Long id)
    {
        return AjaxResult.success(goodsService.selectGoodsById(id));
    }

    /**
     * 新增商品
     */
    @PreAuthorize("@ss.hasPermi('system:goods:add')")
    @Log(title = "商品", businessType = BusinessType.INSERT)
    @PostMapping
    public AjaxResult add(@RequestBody Goods goods)
    {
        return toAjax(goodsService.insertGoods(goods));
    }

    /**
     * 修改商品
     */
    @PreAuthorize("@ss.hasPermi('system:goods:edit')")
    @Log(title = "商品", businessType = BusinessType.UPDATE)
    @PutMapping
    public AjaxResult edit(@RequestBody Goods goods)
    {
        return toAjax(goodsService.updateGoods(goods));
    }

    /**
     * 删除商品
     */
    @PreAuthorize("@ss.hasPermi('system:goods:remove')")
    @Log(title = "商品", businessType = BusinessType.DELETE)
    @DeleteMapping("/remove")
    public AjaxResult remove(@RequestParam("goodsId") Long goodsId, @RequestParam(required = false) boolean boo) {
        return AjaxResult.success(goodsService.deleteGoodsById(goodsId, boo));
    }

    /**
     * 根据商品id获取sku列表
     */
    @PreAuthorize("@ss.hasPermi('system:goods:sku:list')")
    @GetMapping("/sku/list/{goodsId}")
    @ApiOperation("根据商品id获取sku列表")
    public AjaxResult getGoodsSkuList(@PathVariable("goodsId") Long goodsId) {
        return AjaxResult.success(goodsService.getGoodsSkus(goodsId));
    }

    /**
     * 删除sku
     */
    @PreAuthorize("@ss.hasPermi('system:goods:sku:delete')")
    @GetMapping("/sku/delete/{id}")
    @ApiOperation("删除sku")
    public AjaxResult deleteSkuManual(@PathVariable("id") Long skuId) {
        GoodsSku sku = goodsSkuService.selectGoodsSkuById(skuId);
        Assert.notNull(sku, "未查询到匹配的sku记录[id=" + skuId + "]");
        Assert.isTrue(Objects.equals(sku.getIsDeleted(), SkuDeleteState.NORMAL.getState()), "当前状态不允许做该操作");
        Goods goods = goodsService.selectGoodsById(sku.getGoodsId());
        Assert.notNull(goods, "未查询到sku匹配的商品信息");
        Assert.isTrue(Objects.equals(goods.getState(), "4"), "需先下架商品之后才可删除");
        sku.setIsDeleted(SkuDeleteState.DELETE_MANUAL.getState());
        goodsSkuService.updateGoodsSku(sku);
        return AjaxResult.success();
    }

    /**
     * 恢复sku
     */
    @PreAuthorize("@ss.hasPermi('system:goods:sku:recover')")
    @GetMapping("/sku/recover/{id}")
    @ApiOperation("恢复sku")
    public AjaxResult recoverSku(@PathVariable("id") Long skuId) {
        GoodsSku sku = goodsSkuService.selectGoodsSkuById(skuId);
        Assert.notNull(sku, "未查询到匹配的sku记录[id=" + skuId + "]");
        Assert.isTrue(Objects.equals(sku.getIsDeleted(), SkuDeleteState.DELETE_MANUAL.getState()), "当前状态不允许做该操作");
        sku.setIsDeleted(SkuDeleteState.NORMAL.getState());
        goodsSkuService.updateGoodsSku(sku);
        return AjaxResult.success();
    }
}
