package com.soss.system.service.impl;

import com.soss.common.core.domain.AjaxResult;
import com.soss.common.enums.RefundState;
import com.soss.common.exception.ServiceException;
import com.soss.common.utils.GenerateCode;
import com.soss.common.utils.StringUtils;
import com.soss.system.constants.OrderStatusConstant;
import com.soss.system.domain.Order;
import com.soss.system.domain.OrderDetail;
import com.soss.system.domain.OrderRefund;
import com.soss.system.domain.vo.OrderQuery;
import com.soss.system.mapper.OrderDetailMapper;
import com.soss.system.mapper.OrderMapper;
import com.soss.system.mapper.OrderRefundMapper;
import com.soss.system.service.IOrderRefundService;
import io.jsonwebtoken.lang.Assert;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.Arrays;
import java.util.Date;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * 订单退款Service业务层处理
 *
 * @author zxq
 * @date 2022-04-28
 */
@Service
@Slf4j
public class OrderRefundServiceImpl implements IOrderRefundService {
    @Autowired
    private OrderRefundMapper orderRefundMapper;
    @Autowired
    private OrderServiceImpl orderService;
    @Autowired
    private OrderDetailMapper orderDetailMapper;
    @Autowired
    private OrderMapper orderMapper;
    @Value("${weixin.refund-no-prefix:R}")
    private String refundNoPrefix;

    /**
     * 查询订单退款
     *
     * @param id 订单退款主键
     * @return 订单退款
     */
    @Override
    public OrderRefund selectOrderRefundById(String id) {
        OrderRefund orderRefund = orderRefundMapper.selectOrderRefundById(id);
        Assert.notNull(orderRefund, "未查询到匹配的退款记录");
        this.wrapperOrderInfo(orderRefund);
        return orderRefund;
    }

    private void wrapperOrderInfo(OrderRefund orderRefund) {
        Order order = orderService.selectOrderById(orderRefund.getOrderId());
        OrderDetail detail = new OrderDetail();
        detail.setRefundId(orderRefund.getId());
        List<OrderDetail> orderDetails = orderDetailMapper.selectOrderDetailList(detail);
        order.setOrderDetails(orderDetails);
        orderRefund.setOrder(order);
    }

    /**
     * 查询订单退款列表
     *
     * @param orderRefund 订单退款
     * @return 订单退款
     */
    @Override
    public List<OrderRefund> selectOrderRefundList(OrderRefund orderRefund) {
        List<OrderRefund> orderRefunds = orderRefundMapper.selectOrderRefundList(orderRefund);
        for (OrderRefund refund : orderRefunds) {
            OrderDetail detail = new OrderDetail();
            detail.setRefundId(refund.getId());
            List<OrderDetail> orderDetails = orderDetailMapper.selectOrderDetailList(detail);
            refund.setOrderDetails(orderDetails);
        }
        return orderRefunds;
    }

    /**
     * 新增订单退款
     *
     * @param orderRefund 订单退款
     * @return 结果
     */
    @Override
    @Transactional
    public AjaxResult insertOrderRefund(OrderRefund orderRefund) {
        Order order = orderService.selectById(orderRefund.getOrderId());
        if (order == null) {
            return AjaxResult.error("找不到该订单");
        }

        String state = order.getState();
        // 制作中、制作完成、取餐中、已完成、待取超时、退款失败、部分退款、制作中取消
        List<String> status = Arrays.asList(OrderStatusConstant.production, OrderStatusConstant.productionCompleted,
                OrderStatusConstant.Taking, OrderStatusConstant.completed, OrderStatusConstant.timeout,
                OrderStatusConstant.refundFailed, OrderStatusConstant.PartialRefund, OrderStatusConstant.productionCancel);
        if (OrderStatusConstant.Paid.equals(state)) {
            //orderRefund.setRefundAmount(order.getAmount());
            // 4 支付制作中 5 制作完成未取 6 取餐中 7 取餐完成
        } else if (status.contains(state)) {
            String desc = orderRefund.getDesc();
            if (StringUtils.isEmpty(desc)) {
                return AjaxResult.error("请联系客服进行退款");
            }
        } else {
            return AjaxResult.error("该订单状态不允许退款");
        }

        BigDecimal orderDetailAmount;
        if (!CollectionUtils.isEmpty(orderRefund.getOrderDetailIds())) {
            List<OrderDetail> orderDetails = orderDetailMapper.listDetailByIds(orderRefund.getOrderDetailIds());
            List<String> illegalGoodNames = orderDetails.stream().filter(orderDetail -> !Objects.equals(orderDetail.getOrderId(), order.getId()))
                    .map(OrderDetail::getGoodsName).collect(Collectors.toList());
            if (!CollectionUtils.isEmpty(illegalGoodNames)) {
                return AjaxResult.error("商品" + illegalGoodNames + "和订单不匹配");
            }
            illegalGoodNames = orderDetails.stream().filter(orderDetail -> orderDetail.getRefundId() != null)
                    .map(OrderDetail::getGoodsName).collect(Collectors.toList());
            if (!CollectionUtils.isEmpty(illegalGoodNames)) {
                return AjaxResult.error("商品" + illegalGoodNames + "已发起过退款");
            }
            orderDetailAmount = orderDetails.stream().map(OrderDetail::getRealAmount).reduce(BigDecimal.ZERO, BigDecimal::add);
            if (orderDetailAmount.compareTo(orderRefund.getRefundAmount()) != 0) {
                log.warn("退款金额[{}]与商品明细总额[{}]不符", orderRefund.getRefundAmount(), orderDetailAmount);
                return AjaxResult.error("退款金额与商品明细总额不符");
            }
        } else {
            OrderDetail detail = new OrderDetail();
            detail.setOrderId(order.getId());
            List<OrderDetail> orderDetails = orderDetailMapper.selectOrderDetailList(detail);
            orderDetailAmount = orderDetails.stream().map(OrderDetail::getRealAmount).reduce(BigDecimal.ZERO, BigDecimal::add);
            List<Long> orderDetailIds = orderDetails.stream().map(OrderDetail::getId).collect(Collectors.toList());
            orderRefund.setOrderDetailIds(orderDetailIds);
        }

        orderRefund.setCreatedAt(new Date());
        orderRefund.setUpdatedAt(new Date());
        orderRefund.setState(RefundState.PENDING.getState());
        orderRefund.setOrderNo(order.getOrderNo());
        orderRefund.setRefundNo(GenerateCode.getCode(refundNoPrefix, "%09d"));
        orderRefund.setTotalFee(orderDetailAmount);
        orderRefund.setOrderNo(order.getOrderNo());
        if (StringUtils.isEmpty(orderRefund.getCreateUserName())) {
            orderRefund.setCreateUserName(order.getUserName());
        }
        orderRefundMapper.insertOrderRefund(orderRefund);
        if (order.getPaidAmount().compareTo(BigDecimal.ZERO) == 0) {
            order.setState(OrderStatusConstant.refund);
        } else {
            order.setState(OrderStatusConstant.refunding);
        }
        orderService.updateOrder(order);
        orderDetailMapper.bindRefund(orderRefund.getId(), orderRefund.getOrderDetailIds());
        orderRefund.setOrder(order);
        return AjaxResult.success(orderRefund);
    }

    /**
     * 修改订单退款
     *
     * @param orderRefund 订单退款
     * @return 结果
     */
    @Override
    @Transactional
    public int updateOrderRefund(OrderRefund orderRefund) {
        if (Objects.equals(orderRefund.getState(), RefundState.FAILURE.getState())) {
            Order order = new Order();
            order.setId(orderRefund.getOrderId());
            order.setState(OrderStatusConstant.refundFailed);
            orderMapper.updateOrder(order);
        }
        return orderRefundMapper.updateOrderRefund(orderRefund);
    }

    /**
     * 批量删除订单退款
     * 
     * @param ids 需要删除的订单退款主键
     * @return 结果
     */
    @Override
    public int deleteOrderRefundByIds(String[] ids)
    {
        return orderRefundMapper.deleteOrderRefundByIds(ids);
    }

    /**
     * 删除订单退款信息
     * 
     * @param id 订单退款主键
     * @return 结果
     */
    @Override
    public int deleteOrderRefundById(String id)
    {
        return orderRefundMapper.deleteOrderRefundById(id);
    }

    @Override
    public OrderRefund getOrderRefund(String id) {
        OrderRefund orderRefund = orderRefundMapper.selectOrderRefundById(id);
        if("0".equals(orderRefund.getState()) ||"3".equals(orderRefund.getState())){

        }else{
            throw new ServiceException("该状态不允许退款");
        }
        Order order = orderService.selectById(orderRefund.getOrderId());
        orderRefund.setTotalFee(order.getAmount());
        return orderRefund;
    }

    @Override
    public List<OrderRefund> selectOrderRefundList(OrderQuery orderQuery) {
        List<OrderRefund> orderRefunds = orderRefundMapper.selectList(orderQuery);
        for (OrderRefund refund : orderRefunds) {
            this.wrapperOrderInfo(refund);
        }
        return orderRefunds;
    }
}
