package com.soss.web.controller.coffee;

import com.soss.common.core.controller.BaseController;
import com.soss.common.core.domain.AjaxResult;
import com.soss.common.core.domain.model.LoginUser;
import com.soss.common.core.page.TableDataInfo;
import com.soss.common.exception.ServiceException;
import com.soss.common.utils.StringUtils;
import com.soss.common.utils.spring.SpringUtils;
import com.soss.framework.web.service.TokenService;
import com.soss.system.domain.Customer;
import com.soss.system.domain.Order;
import com.soss.system.domain.vo.customer.CustomerQueryVo;
import com.soss.system.domain.vo.customer.CustomerResultVo;
import com.soss.system.service.ICustomerService;
import io.jsonwebtoken.lang.Assert;
import io.swagger.annotations.ApiOperation;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletRequest;
import java.util.List;
import java.util.Objects;

/**
 * 用户信息Controller
 *
 * @author zxq
 * @date 2022-05-01
 */
@RestController
@RequestMapping("/system/customer")
public class CustomerController extends BaseController {
    @Autowired
    private ICustomerService customerService;
    @Autowired
    private TokenService tokenService;

    @PostMapping("/allow")
    @ApiOperation("用户授权")
    public AjaxResult allow(HttpServletRequest request, @RequestBody Customer customer) {
        TokenService bean = SpringUtils.getBean(TokenService.class);
        LoginUser loginUser = bean.getLoginUser(request);
        if (loginUser == null) {
            throw new ServiceException("请先登录");
        }
        customer.setId(loginUser.getOpenId());
        Assert.notNull(customer.getSendMsgFlag1(), "用户授权发送消息1未传递");
        Assert.notNull(customer.getSendMsgFlag2(), "用户授权发送消息2未传递");
        Assert.notNull(customer.getSendMsgFlag3(), "用户授权发送消息3未传递");
        Assert.notNull(customer.getRememberFlag(), "记住标识未传递");
        customerService.updateSendMsgFlag(customer);
        return AjaxResult.success();
    }

    @GetMapping("/detail")
    @ApiOperation("查询用户详细信息")
    public AjaxResult updateCust(HttpServletRequest request) {
        LoginUser loginUser = tokenService.getLoginUser(request);
        if (loginUser == null) {
            throw new ServiceException("请先登录");
        }
        Customer customer = customerService.selectCustById(loginUser.getOpenId());
        return AjaxResult.success(customer);
    }

    @PostMapping("/update/avatar-url")
    @ApiOperation("修改用户头像")
    public AjaxResult updateAvatarUrl(HttpServletRequest request, @RequestBody Customer customer) {
        custPmsCheck(request, customer.getId());
        Assert.isTrue(!StringUtils.isEmpty(customer.getAvatarUrl()), "用户头像未传递");
        Assert.isNull(customer.getBirthday(), "该方法不支持修改生日");
        customerService.updateCustomer(customer);
        return AjaxResult.success();
    }

    @PostMapping("/update/base")
    @ApiOperation("修改用户基本信息：昵称和性别")
    public AjaxResult updateCust(HttpServletRequest request, @RequestBody Customer customer) {
        custPmsCheck(request, customer.getId());
        Assert.isTrue(!StringUtils.isEmpty(customer.getUserName()), "昵称未传递");
        Assert.isTrue(customer.getUserName().length() <= 7, "昵称最多七个字");
        Assert.notNull(customer.getSex(), "性别还未传递");
        Assert.isNull(customer.getBirthday(), "该方法不支持修改生日");
        customerService.updateCustomer(customer);
        return AjaxResult.success();
    }

    @PostMapping("/update/birth")
    @ApiOperation("修改用户生日")
    public AjaxResult updateCustBirth(HttpServletRequest request, @RequestBody Customer customer) {
        custPmsCheck(request, customer.getId());
        Assert.notNull(customer.getBirthday(), "生日还未传递");
        customerService.updateCustBirthday(customer);
        return AjaxResult.success();
    }

    private void custPmsCheck(HttpServletRequest request, String custId) {
        Assert.isTrue(!StringUtils.isEmpty(custId), "用户id未传递");
        LoginUser loginUser = tokenService.getLoginUser(request);
        if (loginUser == null) {
            throw new ServiceException("请先登录");
        }
        Assert.isTrue(Objects.equals(custId, loginUser.getOpenId()), "不能修改他人信息");
    }

    /**
     * 查询用户信息列表
     */
    @PreAuthorize("@ss.hasPermi('system:customer:list')")
    @GetMapping("/list")
    public TableDataInfo list(CustomerQueryVo customerQueryVo) {
        startPage();
        List<CustomerResultVo> list = customerService.selectCustomerList(customerQueryVo);
        return getDataTable(list);
    }

    /**
     * 获取用户信息详细信息
     */
    @PreAuthorize("@ss.hasPermi('system:customer:query')")
    @GetMapping
    public TableDataInfo getInfo(@RequestParam("id") String id, @RequestParam String status) {
        startPage();
        List<Order> orders = customerService.selectCustomerById(id, status);
        return getDataTable(orders);
    }

    @PreAuthorize("@ss.hasPermi('system:customer:order-simple-stat')")
    @GetMapping("/order-simple-stat/{custId}")
    @ApiOperation("查询用户的下单基本信息：下单总量和下单总额")
    public AjaxResult getCustOrderSimpleStat(@PathVariable String custId) {
        return AjaxResult.success(customerService.getCustSimpleOrderStat(custId));
    }
}
