package com.soss.system.service.impl;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.soss.common.enums.SkuDeleteState;
import com.soss.common.exception.ServiceException;
import com.soss.common.utils.GenerateCode;
import com.soss.common.utils.StringUtils;
import com.soss.system.domain.*;
import com.soss.system.push.impl.PushServiceImpl;
import com.soss.system.mapper.*;
import com.soss.system.service.IGoodsService;
import jodd.util.StringPool;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 商品Service业务层处理
 *
 * @author zxq
 * @date 2022-04-28
 */
@Service
@Slf4j
public class GoodsServiceImpl implements IGoodsService {
    @Autowired
    private GoodsMapper goodsMapper;
    @Autowired
    private GoodsSkuMapper goodsSkuMapper;
    @Autowired
    private GoodsTagMapper goodsTagMapper;
    @Autowired
    private ShopServiceImpl shopService;
    @Autowired
    private ShopGoodsMapper shopGoodsMapper;
    @Autowired
    private ShopGoodsSkuMapper shopGoodsSkuMapper;
    @Autowired
    private OrderMapper orderMapper;
    @Autowired
    private OrderDetailMapper orderDetailMapper;
    @Autowired
    private GoodsCategoryMapper goodsCategoryMapper;
    @Autowired
    private ShopRecommendMapper shopRecommendMapper;
    @Autowired
    private MachineMapper machineMapper;
    @Autowired
    private PushServiceImpl pushService;

    /**
     * 查询商品
     *
     * @param id 商品主键
     * @return 商品
     */
    @Override
    public Goods selectGoodsById(Long id) {
        Goods goods = goodsMapper.selectGoodsById(id);
        Assert.notNull(goods, "未查询到匹配的商品信息");
        resolverCategoryName(Collections.singletonList(goods));
        GoodsTag goodsTag = new GoodsTag();
        goodsTag.setGoodsId(goods.getId());
        goods.setGoodsTagList(goodsTagMapper.selectGoodsTagList(goodsTag).stream().map(GoodsTag::getTag).collect(Collectors.toList()));
        return goods;
    }

    @Override
    public List<Goods> selectGoodsByIds(List<Long> ids) {
        Assert.notEmpty(ids, "goods ids cannot be empty");
        List<Goods> goods = goodsMapper.selectGoodsByIds(ids);
        this.resolverCategoryName(goods);
        return goods;
    }

    /**
     * 查询商品列表
     *
     * @param goods 商品
     * @return 商品
     */
    @Override
    public List<Goods> selectGoodsList(Goods goods) {
        List<Goods> goodsList = goodsMapper.selectGoodsList(goods);
        this.resolverCategoryName(goodsList);
        this.resolverGoodsVolume(null, goodsList);
        /*for (Goods good : goodsList) {
            int count = orderMapper.selectSalesVolume(null, good.getId());
            good.setSalesVolume(count);
        }*/
        return goodsList;
    }

    @Override
    public void resolverGoodsVolume(Long shopId, List<Goods> goodsList) {
        if (CollectionUtils.isEmpty(goodsList)) {
            return;
        }

        /** 销量解析 */
        List<Long> goodsIds = goodsList.stream().map(Goods::getId).collect(Collectors.toList());
        List<OrderDetail> orderDetails = orderDetailMapper.selectGoodsSalesVolume(shopId, goodsIds);
        Map<Long, Integer> goodsVolumeMap = orderDetails.stream().collect(Collectors.toMap(OrderDetail::getGoodsId, OrderDetail::getNum));

        goodsList.forEach(goods -> goods.setSalesVolume(goodsVolumeMap.containsKey(goods.getId()) ? goodsVolumeMap.get(goods.getId()) : 0));
    }

    @Override
    public void resolverCategoryName(List<Goods> goodsList) {
        if (CollectionUtils.isEmpty(goodsList)) {
            return;
        }

        /** 品类解析 */
        List<Long> categoryIds = new ArrayList<>(16);
        goodsList.forEach(good -> {
            String[] categoryIdArr = good.getCategory().split(StringPool.COMMA);
            List<Long> goodCategoryIds = new ArrayList<>();
            for (String categoryId : categoryIdArr) {
                if (!StringUtils.isEmpty(categoryId)) {
                    goodCategoryIds.add(Long.parseLong(categoryId));
                }
            }
            good.setCategoryIds(goodCategoryIds);
            categoryIds.addAll(goodCategoryIds);
        });
        List<GoodsCategory> categories = goodsCategoryMapper.selectGoodsCategoryByIds(categoryIds);
        Map<Long, String> categoryMap = categories.stream().collect(Collectors.toMap(GoodsCategory::getId, GoodsCategory::getName));

        goodsList.forEach(good -> {
            String categoryName = Arrays.stream(good.getCategory().split(StringPool.COMMA)).filter(StringUtils::isNotEmpty)
                    .map(Long::parseLong).map(categoryMap::get)
                    .collect(Collectors.joining(StringPool.COMMA));
            good.setCategoryName(categoryName);
        });
    }

    /**
     * 新增商品
     *
     * @param goods 商品
     * @return 结果
     */
    @Override
    @Transactional
    public int insertGoods(Goods goods) {
        if (goods.getDiscount().compareTo(goods.getPrice()) > 0) {
            throw new ServiceException("折扣价不能大于原价");
        }
        Assert.notEmpty(goods.getCategoryIds(), "商品分类还未传递");
        goods.setCategory(transferCategoryId(goods.getCategoryIds()));
        goods.setCode(GenerateCode.getCode("G"));
        goods.setCreatedAt(new Date());
        goods.setUpdatedAt(new Date());
        goods.setIsDeleted(0L);
        goods.setState("1");
        goodsMapper.insertGoods(goods);
        insertgoodsExt(goods);
        insertGoodsTags(goods);
        return 1;
    }

    private String transferCategoryId(List<Long> categoryIds) {
        String categoryIdStr = categoryIds.stream().map(String::valueOf).collect(Collectors.joining(","));
        return StringPool.COMMA + categoryIdStr + StringPool.COMMA;
    }

    private void insertGoodsTags(Goods goods) {
        List<String> goodsTagList = goods.getGoodsTagList();
        if (CollectionUtils.isEmpty(goodsTagList)) {
            return;
        }
        for (String tagName : goodsTagList) {
            GoodsTag goodsTag = new GoodsTag();
            goodsTag.setGoodsId(goods.getId());
            goodsTag.setState("1");
            goodsTag.setCreatedAt(new Date());
            goodsTag.setTag(tagName);
            goodsTagMapper.insertGoodsTag(goodsTag);
        }
    }

    private void insertgoodsExt(Goods goods) {
        String specString = goods.getSpec();
        long goodsId = goods.getId();
        if (StringUtils.isNotEmpty(specString)) {
            List<Spec> specs = JSONArray.parseArray(specString, Spec.class);
            List<RuleNode> ruleNodes = new ArrayList<>();
            List<RuleNode> ruleNodeList = new ArrayList<>();
            for (int i = 0; i < specs.size(); i++) {
                Spec spec = specs.get(i);
                List<SpecRule> specRules = spec.getSpecRules();
                if (specRules != null && !specRules.isEmpty()) {
                    if (ruleNodes.isEmpty()) {
                        for (SpecRule specRule : specRules) {
                            RuleNode<SpecRule> ruleNode = new RuleNode<>();
                            ruleNode.setRuleNode(specRule);
                            ruleNodes.add(ruleNode);
                            ruleNodeList.add(ruleNode);
                        }
                    } else {
                        ruleNodeList.clear();
                        for (RuleNode<SpecRule> ruleNode : ruleNodes) {
                            putRuleNodeChild(ruleNode, specRules, ruleNodeList);
                        }
                    }
                }
            }
            createSku(ruleNodeList, goodsId, goods.getDiscount(), goods.getPrice());
        }
    }

    private void createSku(List<RuleNode> ruleNodeList, long goodsId, BigDecimal discount, BigDecimal price) {

        for (RuleNode<SpecRule> ruleNode : ruleNodeList) {
            List<SpecRule> specRuleList = new ArrayList<>();

            RuleNode<SpecRule> parentRuleNode = ruleNode.getParentRuleNode();
            specRuleList.add(ruleNode.getRuleNode());
            if (parentRuleNode != null) {
                findPartentRuleNode(parentRuleNode, specRuleList);
            }
            Collections.reverse(specRuleList);
            GoodsSku goodsSku = new GoodsSku();
            goodsSku.setGoodsId(goodsId);
            goodsSku.setCreatedAt(new Date());
            goodsSku.setIsDeleted(SkuDeleteState.NORMAL.getState());
            goodsSku.setUpdatedAt(new Date());
            BigDecimal price1 = new BigDecimal(price.toPlainString());
            BigDecimal discount1 = new BigDecimal(discount.toPlainString());
            log.info("商品价格为：{}，{}", price.toPlainString(), discount.toPlainString());
            for (SpecRule specRule : specRuleList) {
                price1 = price1.add(specRule.getAmount());
                discount1 = discount1.add(specRule.getAmount());
            }
            log.info("产品价格为：{}，{}", price.toPlainString(), discount.toPlainString());
            goodsSku.setPrice(price1);
            goodsSku.setDiscount(discount1);
            goodsSku.setRuleList(JSONArray.toJSONString(specRuleList));
            goodsSkuMapper.insertGoodsSku(goodsSku);
        }
    }

    private void findPartentRuleNode(RuleNode<SpecRule> ruleRuleNode, List<SpecRule> specRuleList) {
        specRuleList.add(ruleRuleNode.getRuleNode());
        RuleNode<SpecRule> parentRuleNode = ruleRuleNode.getParentRuleNode();
        if (parentRuleNode != null) {
            findPartentRuleNode(parentRuleNode, specRuleList);
        }
    }


    private void putRuleNodeChild(RuleNode ruleNode, List<SpecRule> specRules, List<RuleNode> ruleNodeList) {
        List<RuleNode> childNode = ruleNode.getChildNode();
        if (childNode == null || childNode.isEmpty()) {
            List<RuleNode> ruleNodes = new ArrayList<>();
            ruleNode.setChildNode(ruleNodes);
            for (SpecRule specRule : specRules) {
                RuleNode<SpecRule> ruleNode1 = new RuleNode<>();
                ruleNode1.setRuleNode(specRule);
                ruleNode1.setParentRuleNode(ruleNode);
                ruleNodes.add(ruleNode1);
                ruleNodeList.add(ruleNode1);
            }
        } else {
            for (RuleNode node : childNode) {
                putRuleNodeChild(node, specRules, ruleNodeList);
            }
        }

    }

    /**
     * 修改商品
     *
     * @param goods 商品
     * @return 结果
     */
    @Override
    public int updateGoods(Goods goods) {
        if (!CollectionUtils.isEmpty(goods.getCategoryIds())) {
            goods.setCategory(this.transferCategoryId(goods.getCategoryIds()));
        }
        Goods goodsOri = goodsMapper.selectGoodsById(goods.getId());
        int i = goodsMapper.updateGoods(goods);

        goodsTagMapper.deleteGoodsTagByGoodsId(String.valueOf(goods.getId()));
        insertGoodsTags(goods);

        if ((StringUtils.isEmpty(goods.getSpec()) || Objects.equals(goodsOri.getSpec(), goods.getSpec()))) {
            BigDecimal discountDiff = goods.getDiscount().subtract(goodsOri.getDiscount());
            BigDecimal priceDiff = goods.getPrice().subtract(goodsOri.getPrice());
            if (discountDiff.compareTo(BigDecimal.ZERO) != 0 || priceDiff.compareTo(BigDecimal.ZERO) != 0) {
                goodsSkuMapper.updateSkuPrice(goods.getId(), discountDiff, priceDiff);
            }
            return i;
        }
        /*if(StringUtils.isNotEmpty(goods.getSpec())){
            goodsSkuMapper.deleteGoodsSkuByGoodsId(String.valueOf(goods.getId()));
        }*/
        goodsSkuMapper.deleteGoodsSkuByGoodsId(String.valueOf(goods.getId()));
        insertgoodsExt(goods);
        return i;
    }

    /**
     * 批量删除商品
     *
     * @param ids 需要删除的商品主键
     * @return 结果
     */
    @Override
    public int deleteGoodsByIds(String[] ids) {
        return goodsMapper.deleteGoodsByIds(ids);
    }

    /**
     * 删除商品信息
     *
     * @param id 商品主键
     * @return 结果
     */
    @Override
    public int deleteGoodsById(String id) {
        return 1;
    }

    @Override
    public String putShop(Long goodsId, boolean boo) {
        GoodsSku goodsSku = new GoodsSku();
        goodsSku.setGoodsId(goodsId);
        List<GoodsSku> goodsSkus = goodsSkuMapper.selectGoodsSkuList(goodsSku);
        if (goodsSkus.isEmpty()) {
            throw new ServiceException("商品信息不全，请修改商品");
        }
        Goods goods = new Goods();
        goods.setId(goodsId);
        goods.setState("3");
        goods.setUpdatedAt(new Date());
        goods.setShelfAt(new Date());
        int i = goodsMapper.updateGoods(goods);
        if (boo && i > 0) {
            Shop shop = new Shop();
            List<Shop> shops = shopService.selectShopList(shop);
            for (Shop shop1 : shops) {
                if (!Objects.equals(3, shop1.getState())) {
                    try {
                        shopService.addGoods(shop1.getId(), goodsId);
                    } catch (ServiceException serviceException) {
                    }
                }
            }
        }
        return null;
    }

    @Override
    public String offShop(Long goodsId, boolean boo) {
        ShopGoods shopGoods = new ShopGoods();
        shopGoods.setGoodsId(goodsId);
        List<ShopGoods> shopGoodsList = shopGoodsMapper.selectShopGoodsList(shopGoods);
        if (!boo) {
            Boolean isExist = shopGoodsList.size() > 0;
            if (isExist) {
                return "99";
            }
        }
        Goods goods = new Goods();
        goods.setId(goodsId);
        goods.setUpdatedAt(new Date());
        goods.setState("4");
        int i = goodsMapper.updateGoods(goods);
        for (ShopGoods shopGoods1 : shopGoodsList) {
            shopGoodsMapper.deleteShopGoodsById(shopGoods1.getId());
            shopGoodsSkuMapper.deleteShopGoodsSkuByGoodsId(shopGoods1.getGoodsId());
            List<String> goodsIds = Arrays.asList(String.valueOf(shopGoods1.getGoodsId()));
            shopRecommendMapper.deleteByShopIdGoods(shopGoods1.getShopId(), goodsIds);
            updateApplication(shopGoods1.getShopId());
        }

        return i + "";
    }

    public void updateApplication(long shopId) {
        Machine machine = new Machine();
        machine.setShopId(shopId);
        List<Machine> machines = machineMapper.selectMachineList(machine);
        if (!machines.isEmpty()) {
            JSONObject jsonObject = new JSONObject();
            jsonObject.put("action", "GOODS_CHANGED");
            jsonObject.put("timestamp", System.currentTimeMillis());
            pushService.push(machines.get(0).getCode(), JSON.toJSONString(jsonObject));
        }
    }

    @Override
    public String deleteGoodsById(Long id, boolean boo) {
        ShopGoods shopGoods = new ShopGoods();
        shopGoods.setGoodsId(id);
        List<ShopGoods> shopGoodsList = shopGoodsMapper.selectShopGoodsList(shopGoods);
        if (!boo) {
            Boolean isExist = shopGoodsList.size() > 0;
            if (isExist) {
                return "99";
            }
        }
        Goods goods = new Goods();
        goods.setIsDeleted(1L);
        goods.setId(id);
        goods.setState("5");
        goodsMapper.updateGoods(goods);
        goodsSkuMapper.deleteGoodsSkuByGoodsId(String.valueOf(id));
        goodsTagMapper.deleteGoodsTagByGoodsId(String.valueOf(id));
        return "1";
    }

    @Override
    public List<GoodsSku> getGoodsSkus(long goodsId) {
        GoodsSku goodsSku = new GoodsSku();
        goodsSku.setGoodsId(goodsId);
        return goodsSkuMapper.selectGoodsSkuList(goodsSku);
    }

    @Data
    class RuleNode<T> {
        private RuleNode parentRuleNode;
        private T ruleNode;
        private List<RuleNode> childNode;
    }

    @Override
    public void resolverGoodsOfOrder(Order order) {
        if (order == null) {
            return;
        }
        List<OrderDetail> orderDetails = order.getOrderDetails();
        if (CollectionUtils.isEmpty(orderDetails)) {
            return;
        }
        List<Long> goodsIds = orderDetails.stream().map(OrderDetail::getGoodsId).collect(Collectors.toList());
        List<Goods> goods = this.selectGoodsByIds(goodsIds);
        orderDetails.forEach(orderDetail -> goods.forEach(good -> {
            if (Objects.equals(orderDetail.getGoodsId(), good.getId())) {
                orderDetail.setGoods(good);
            }
        }));
    }
}
