package com.soss.system.service.impl;

import com.soss.common.enums.CouponCategoryType;
import com.soss.common.enums.CouponState;
import com.soss.common.exception.ServiceException;
import com.soss.common.utils.GenerateCode;
import com.soss.system.domain.Coupon;
import com.soss.system.domain.CouponCategory;
import com.soss.system.domain.CouponRule;
import com.soss.system.domain.po.CouponPo;
import com.soss.system.mapper.CouponCategoryMapper;
import com.soss.system.mapper.CouponMapper;
import com.soss.system.mapper.CouponRuleMapper;
import com.soss.system.mapper.CouponUserMapper;
import com.soss.system.service.ICouponService;
import com.soss.system.utils.ArrayUtil;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;

import java.math.BigDecimal;
import java.time.LocalDateTime;
import java.util.Arrays;
import java.util.List;
import java.util.stream.Collectors;

/**
 * <p>
 * 优惠券 服务实现类
 * </p>
 *
 * @author caiyt
 * @since 2022-07-21
 */
@Service
public class CouponServiceImpl implements ICouponService {
    @Autowired
    private CouponCategoryMapper couponCategoryMapper;
    @Autowired
    private CouponMapper couponMapper;
    @Autowired
    private CouponRuleMapper couponRuleMapper;
    @Autowired
    private CouponUserMapper couponUserMapper;

    /**
     * 查询优惠券列表
     *
     * @param coupon
     * @return
     */
    @Override
    public List<Coupon> list(Coupon coupon) {
        return couponMapper.listCoupon(coupon);
    }

    @Override
    public Coupon detail(Integer id) {
        Coupon coupon = couponMapper.selectCouponById(id);
        Assert.notNull(coupon, "未查询到匹配的优惠券记录[id=" + id + "]");
        return coupon;
    }

    @Override
    @Transactional
    public void save(CouponPo couponPo) {
        Coupon coupon = new Coupon();
        CouponRule couponRule = new CouponRule();

        this.checkAndWrapperCoupon(couponPo, coupon, couponRule, true);
        couponRule.setState(CouponState.ONLINE.getState());
        couponRuleMapper.insertCouponRule(couponRule);
        coupon.setRuleId(couponRule.getId());
        coupon.setState(CouponState.ONLINE.getState());
        couponMapper.insertCoupon(coupon);
    }

    @Override
    @Transactional
    public void update(CouponPo couponPo) {
        Assert.notNull(couponPo.getId(), "要修改的优惠券ID还未传递");
        Coupon coupon = this.detail(couponPo.getId());
        long couponReceiveCount = couponUserMapper.getCouponReceiveCount(coupon.getId());
        if (couponReceiveCount > 0) {
            throw new ServiceException("优惠券已被领取，无法编辑，只可下线");
        } else if (!coupon.getState().equals(CouponState.OFFLINE.getState())) {
            throw new ServiceException("修改优惠券需先下架优惠券");
        }
        CouponRule couponRule = couponRuleMapper.selectCouponRuleById(coupon.getRuleId());
        checkAndWrapperCoupon(couponPo, coupon, couponRule, false);
        couponRuleMapper.updateCouponRule(couponRule);
        couponMapper.updateCoupon(coupon);
    }

    @Override
    public void updateCheckPre(Integer id) {
        Long couponReceiveCount = couponUserMapper.getCouponReceiveCount(id);
        if (couponReceiveCount > 0) {
            throw new ServiceException("优惠券已被领取，无法编辑，只可下线");
        }
    }

    private void checkAndWrapperCoupon(CouponPo couponPo, Coupon coupon, CouponRule couponRule, boolean saveFlag) {
        Assert.notNull(couponPo.getName(), "优惠券名称还未填写");
        Assert.notNull(couponPo.getCategoryId(), "优惠券类别还未选择");
        CouponCategory couponCategory = couponCategoryMapper.selectCouponCategoryById(couponPo.getCategoryId());
        Assert.notNull(couponCategory, "未查询到匹配的优惠券类别[id=" + couponPo.getCategoryId() + "]");
        String ruleName, ruleDesc; // 规则名称，规则描述
        if (CouponCategoryType.DEDUCTION.getType().equals(couponCategory.getType())) {
            Assert.notNull(couponPo.getPriceDiscount(), "满减价还未设置");
            Assert.notNull(couponPo.getPriceLimit(), "价格门槛还未设置");
            ruleName = "立减" + couponPo.getPriceDiscount() + "元";
            if (couponPo.getPriceLimit().compareTo(BigDecimal.ZERO) == 0) {
                ruleDesc = "无门槛";
            } else {
                ruleDesc = "满" + couponPo.getPriceLimit() + "元";
            }
        } else if (CouponCategoryType.DISCOUNT.getType().equals(couponCategory.getType())) {
            Assert.notNull(couponPo.getPriceDiscount(), "打折数还未设置");
            Assert.notNull(couponPo.getOrderLimit(), "订单限制还未设置");
            couponPo.setPriceLimit(BigDecimal.ZERO);
            ruleName = couponPo.getPriceDiscount() + "折";
            ruleDesc = couponPo.getOrderLimit() ? "单杯" : "整单";
        } else {
            Assert.notNull(couponPo.getPriceLimit(), "价格门槛还未设置");
            Assert.notNull(couponPo.getOrderLimit(), "订单限制还未设置");
            ruleName = "免单";
            if (couponPo.getPriceLimit().compareTo(BigDecimal.ZERO) == 0) {
                ruleDesc = "无门槛";
            } else {
                ruleDesc = "满" + couponPo.getPriceLimit() + "元";
            }
        }
        Assert.notNull(couponPo.getUserLimit(), "用户可领取次数限制还未设置");
        if (couponPo.getDaysLimit() == null) {
            couponPo.setDaysLimit(0);
        }
        Assert.notNull(couponPo.getUseStartTime(), "优惠券的绝对有效时间还未设置");
        Assert.notNull(couponPo.getUseEndTime(), "优惠券的绝对有效时间还未设置");

        if (couponPo.getReceivableTime() != null && couponPo.getReceivableTime().isAfter(couponPo.getUseStartTime())) {
            throw new ServiceException("优惠券的可领取时间要早于或等于优惠券的绝对有效时间的起始时间");
        }
        if (couponPo.getRelativeTime() != null && couponPo.getRelativeTime() > 0) {
            Assert.isTrue(couponPo.getUseStartTime().plusDays(couponPo.getRelativeTime()).isBefore(couponPo.getUseEndTime()),
                    "实际结束时间已超出生命周期，请修改有效时间。");
        }
        Assert.isTrue((couponPo.getCategoryIds() != null && couponPo.getCategoryIds().length > 0)
                || (couponPo.getGoodsIds() != null && couponPo.getGoodsIds().length > 0), "商品范围还未设置");
        if (!ArrayUtil.isEmpty(couponPo.getCategoryIds()) && !ArrayUtil.isEmpty(couponPo.getGoodsIds())) {
            throw new ServiceException("品类范围和商品范围不能重叠选择");
        }
        couponPo.couponUseAreaLimitCheck();
        Assert.notEmpty(couponPo.getWeekLimit(), "每周几的限制还未传递");
        Integer maxWeekVal = Arrays.stream(couponPo.getWeekLimit()).max(Integer::compareTo).get();
        Integer minWeekVal = Arrays.stream(couponPo.getWeekLimit()).min(Integer::compareTo).get();
        Assert.isTrue(minWeekVal >= 0 && maxWeekVal <= 7, "每周几的值超出范围，应在[0,7]之内");
        Assert.notNull(couponPo.getSendMsg(), "是否短信通知还未设置");
        Assert.isTrue(!couponPo.getSendMsg() || couponPo.getMsgId() != null, "短信模板还未选择");

        coupon.setName(couponPo.getName());
        coupon.setSerialNo(GenerateCode.getCode("CP", "%06d"));
        coupon.setCategoryId(couponCategory.getId());
        coupon.setCategoryName(couponCategory.getName());
        coupon.setType(couponCategory.getType());
        LocalDateTime now = LocalDateTime.now();
        if (saveFlag) {
            coupon.setCreatedAt(now);
        }
        coupon.setUpdatedAt(now);

        BeanUtils.copyProperties(couponPo, couponRule, "name");
        couponRule.setName(ruleName);
        couponRule.setDesc(ruleDesc);
        couponRule.setCategoryIds(Arrays.stream(couponPo.getCategoryIds()).map(String::valueOf).collect(Collectors.joining(",")));
        couponRule.setGoodsIds(Arrays.stream(couponPo.getGoodsIds()).map(String::valueOf).collect(Collectors.joining(",")));
        if (couponPo.getProvince() == null || couponPo.getProvince().length == 0) {
            couponRule.setProvince("");
        } else {
            couponRule.setProvince(String.join(",", couponPo.getProvince()));
        }
        if (couponPo.getCity() == null || couponPo.getCity().length == 0) {
            couponRule.setCity("");
        } else {
            couponRule.setCity(String.join(",", couponPo.getCity()));
        }
        if (couponPo.getArea() == null || couponPo.getArea().length == 0) {
            couponRule.setArea("");
        } else {
            couponRule.setArea(String.join(",", couponPo.getArea()));
        }
        if (couponPo.getShopIds() == null || couponPo.getShopIds().length == 0) {
            couponRule.setShopIds("");
        } else {
            couponRule.setShopIds(Arrays.stream(couponPo.getShopIds()).map(String::valueOf).collect(Collectors.joining(",")));
        }
        couponRule.setWeekLimit(Arrays.stream(couponPo.getWeekLimit()).map(String::valueOf).collect(Collectors.joining(",")));
        if (saveFlag) {
            couponRule.setCreatedAt(now);
        }
        couponRule.setUpdatedAt(now);
    }


    /**
     * 上架优惠券
     *
     * @param id
     * @return
     */
    @Override
    @Transactional
    public int onlineCoupon(Integer id) {
        Coupon online = new Coupon(id, CouponState.ONLINE.getState());
        int updateCnt = couponMapper.updateCoupon(online);
        return updateCnt;
    }

    /**
     * 下架优惠券
     *
     * @param id
     * @return
     */
    @Override
    @Transactional
    public int offlineCoupon(Integer id) {
        Coupon offline = new Coupon(id, CouponState.OFFLINE.getState());
        int updateCnt = couponMapper.updateCoupon(offline);
        return updateCnt;
    }

    /**
     * 删除优惠券
     *
     * @param id
     * @return
     */
    @Override
    @Transactional
    public int deleteCoupon(Integer id) {
        Coupon coupon = this.detail(id);
        if (!coupon.getState().equals(CouponState.OFFLINE.getState()) && !coupon.getState().equals(CouponState.EXPIRED.getState())) {
            throw new ServiceException("[已下线]或[已失效]的优惠券才能删除");
        }
        Coupon delete = new Coupon(id, CouponState.DELETE.getState());
        int updateCnt = couponMapper.updateCoupon(delete);
        return updateCnt;
    }


    /**
     * 获取当前有效的规则集合
     *
     * @return
     */
    @Override
    public List<Coupon> getEffectiveCoupon() {
        return couponMapper.getEffectiveCoupon();
    }
}
