package com.soss.system.service.impl;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.soss.common.core.domain.AjaxResult;
import com.soss.common.enums.MachineState;
import com.soss.common.enums.ShopState;
import com.soss.common.enums.SkuDeleteState;
import com.soss.common.exception.ServiceException;
import com.soss.common.utils.StringUtils;
import com.soss.system.domain.*;
import com.soss.system.domain.vo.SkuCountVo;
import com.soss.system.domain.vo.orderTaking.*;
import com.soss.system.jiguang.impl.JiGuangPushServiceImpl;
import com.soss.system.mapper.*;
import com.soss.system.service.IShopService;
import com.soss.system.utils.ArrayUtil;
import jodd.util.StringPool;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;

import java.util.*;
import java.util.stream.Collectors;

/**
 * 店铺Service业务层处理
 *
 * @author zxq
 * @date 2022-04-28
 */
@Service
@Slf4j
public class ShopServiceImpl implements IShopService {
    @Autowired
    private ShopMapper shopMapper;
    @Autowired
    private MachineMapper machineMapper;
    @Autowired
    private GoodsSkuMapper goodsSkuMapper;
    @Autowired
    private ShopGoodsMapper shopGoodsMapper;
    @Autowired
    private ShopGoodsSkuMapper shopGoodsSkuMapper;
    @Autowired
    private GoodsCategoryMapper goodsCategoryMapper;
    @Autowired
    private OrderMapper orderMapper;
    @Autowired
    private GoodsMapper goodsMapper;
    @Autowired
    private ShopRecommendMapper shopRecommendMapper;
    @Autowired
    private JiGuangPushServiceImpl jiGuangPushService;
    @Autowired
    private GoodsTagMapper goodsTagMapper;
    @Autowired
    private ShopGoodsServiceImpl shopGoodsService;

    /**
     * 查询店铺
     *
     * @param id 店铺主键
     * @return 店铺
     */
    @Override
    public Shop selectShopById(Long id) {
        Shop shop = shopMapper.selectShopById(id);
        if (shop.getDistanceLimit() != null && shop.getDistanceLimit().equals(0.0)) {
            shop.setDistanceLimit(null);
        }
        return shop;
    }

    /**
     * 查询店铺列表
     *
     * @param shop 店铺
     * @return 店铺
     */
    @Override
    public List<Shop> selectShopList(Shop shop) {
        List<Shop> shops = shopMapper.selectShopList(shop);
        for (Shop shop1 : shops) {
            Machine machine = new Machine();
            machine.setShopId(shop1.getId());
            shop1.setMachineCount(machineMapper.selectMachineList(machine).size());
            shop1.setSalesAmount(orderMapper.selectSalesAmount(shop1.getId()));
            shop1.setSalesVolume(orderMapper.selectSalesVolume(shop1.getId(), null));
            if (shop1.getDistanceLimit() != null && shop1.getDistanceLimit().equals(0.0)) {
                shop1.setDistanceLimit(null);
            }
            //shop1.setMachieStatus(machineMapper.selectMachineList(machine).get(0).getState());
        }

        return shops;
    }

    @Override
    public void testStart(Long shopId) {
        Shop shop = shopMapper.selectShopById(shopId);
        Assert.notNull(shop, "未查询到匹配的店铺信息[id=" + shopId + "]");
        if (!Objects.equals(shop.getState(), ShopState.CLOSE.getState())) {
            throw new ServiceException("需暂停营业状态才可以开始测试");
        }
        shopMapper.updateShop(new Shop(shopId, ShopState.TESTING.getState()));
    }

    @Override
    public void testFinish(Long shopId) {
        List<Machine> machines = machineMapper.selectMachineByShopId(shopId);
        if (CollectionUtils.isEmpty(machines)) {
            throw new ServiceException("需绑定机器后才可完成测试");
        }
        // 完成测试后的校验
        long unFinishTestMachineCount = machines.stream().filter(machine -> !machine.getState().equals(MachineState.RUNNING.getState())
                && !Objects.equals(machine.getState(), MachineState.STANDBY.getState())).count();
        if (unFinishTestMachineCount > 0) {
            throw new ServiceException("需机器完成运营测试后才可完成测试");
        }
        Shop shop = new Shop(shopId, ShopState.CLOSE.getState());
        shopMapper.updateShop(shop);
        machines.forEach(machine -> {
            machine.setState(MachineState.RUNNING.getState());
            machineMapper.updateMachine(machine);
        });
    }

    @Override
    public void open(Long shopId) {
        Shop shop = shopMapper.selectShopById(shopId);
        Assert.notNull(shop, "未查询到匹配的店铺信息[id=" + shopId + "]");
        if (!Objects.equals(shop.getState(), ShopState.CLOSE.getState())) {
            throw new ServiceException("店铺暂停营业状态下才能正常营业");
        }
        shopMapper.updateShop(new Shop(shopId, ShopState.OPEN.getState()));
    }

    /**
     * 新增店铺
     *
     * @param shop 店铺
     * @return 结果
     */
    @Override
    public long insertShop(Shop shop) {
        shop.setCreatedAt(new Date());
        shop.setUpdatedAt(new Date());
        shop.setIsDefault(0L);
        shop.setState(ShopState.TESTING.getState());
        shopMapper.insertShop(shop);
        return shop.getId();
    }

    /**
     * 修改店铺
     *
     * @param shop 店铺
     * @return 结果
     */
    @Override
    public int updateShop(Shop shop) {
        shop.setUpdatedAt(new Date());
        if (1 == shop.getIsDefault()) {
            shopMapper.updateNoDefault();
        }
        if (shop.getDistanceLimit() == null) {
            shop.setDistanceLimit(0.0);
        }
        return shopMapper.updateShop(shop);
    }

    /**
     * 批量删除店铺
     *
     * @param ids 需要删除的店铺主键
     * @return 结果
     */
    @Override
    public int deleteShopByIds(String[] ids) {
        for (String id : ids) {
            Shop shop = new Shop();
            shop.setId(Long.parseLong(id));
            shop.setState(ShopState.DELETE.getState());
            shopMapper.updateShop(shop);
        }
        return 1;
    }

    @Override
    @Transactional
    public String addGoods(long shopId, String goodsIds) {
        String[] goodsIdString = goodsIds.split(",");
        for (String s : goodsIdString) {
            Long goodsId = Long.parseLong(s);
            addGoods(shopId, goodsId);

        }
        return null;
    }

    @Transactional
    public String addGoods(long shopId, Long goodsId) {
        ShopGoods shopGoods = new ShopGoods();
        shopGoods.setGoodsId(goodsId);
        shopGoods.setShopId(shopId);
        shopGoods.setUpdatedAt(new Date());
        shopGoods.setCreatedAt(new Date());
        shopGoodsMapper.insertShopGoods(shopGoods);
        shopGoods.setTurn(shopGoods.getId());
        shopGoodsMapper.updateShopGoods(shopGoods);
        GoodsSku goodsSku = new GoodsSku();
        goodsSku.setGoodsId(goodsId);
        goodsSku.setIsDeleted(SkuDeleteState.NORMAL.getState());
        List<GoodsSku> goodsSkus = goodsSkuMapper.selectGoodsSkuList(goodsSku);
        Machine machine = new Machine();
        machine.setShopId(shopId);
        List<Machine> machines = machineMapper.selectMachineList(machine);
        if (machines == null || machines.isEmpty()) {
            throw new ServiceException("请先绑定机器");
        }
        for (Machine machine1 : machines) {
            for (GoodsSku skus : goodsSkus) {
                ShopGoodsSku shopGoodsSku = new ShopGoodsSku();
                shopGoodsSku.setGoodsId(goodsId);
                shopGoodsSku.setSkuId(skus.getId());
                shopGoodsSku.setShopId(shopId);
                shopGoodsSku.setState("1");
                shopGoodsSku.setIsDeleted("0");
                shopGoodsSku.setCreatedAt(new Date());
                shopGoodsSku.setUpdatedAt(new Date());
                shopGoodsSku.setOrigDiscount(skus.getDiscount());
                shopGoodsSku.setDiscount(skus.getDiscount());
                shopGoodsSkuMapper.insertShopGoodsSku(shopGoodsSku);
            }
            updateApplication(Integer.parseInt(machine1.getId()));
        }
        return null;
    }

    public void updateApplication(Integer machineId) {
        JSONObject jsonObject = new JSONObject();
        jsonObject.put("action", "GOODS_CHANGED");
        jsonObject.put("timestamp", System.currentTimeMillis());
        jiGuangPushService.push(machineId, null, jsonObject);
    }

    @Override
    public List<GoodsCategory> getCategoryGoods(long shopId) {
        List<String> categoryIdStrList = goodsCategoryMapper.selectCategoryIdByShopId(shopId);
        if (CollectionUtils.isEmpty(categoryIdStrList)) {
            return new ArrayList<>();
        }
        List<Long> categoryIds = new ArrayList<>(16);
        categoryIdStrList.forEach(categoryIdStr -> {
            List<Long> categoryIdsUnit = Arrays.stream(categoryIdStr.split(StringPool.COMMA)).filter(StringUtils::isNotEmpty).map(Long::parseLong).collect(Collectors.toList());
            categoryIds.addAll(categoryIdsUnit);
        });
        List<GoodsCategory> shopCategories = goodsCategoryMapper.selectGoodsCategoryByIds(categoryIds);
        for (GoodsCategory category : shopCategories) {
            List<Goods> goodsList = shopGoodsMapper.selectShopCategoryGoodsByCategoryId(shopId, category.getId());

            for (Goods goods : goodsList) {
                ShopGoodsSku shopGoodsSku = new ShopGoodsSku();
                shopGoodsSku.setShopId(shopId);
                shopGoodsSku.setGoodsId(goods.getId());
                List<ShopGoodsSku> shopGoodsSkus = shopGoodsSkuMapper.selectShopGoodsSkuList(shopGoodsSku);
                if (!shopGoodsSkus.isEmpty()) {
                    boolean boo = true;
                    for (ShopGoodsSku goodsSkus : shopGoodsSkus) {
                        if ("2".equals(goodsSkus.getState())) {
                            boo = false;
                            break;
                        }
                    }
                    if (!boo) {
                        goods.setState("2");
                    }
                }

                SkuCountVo skuCountVo = shopGoodsSkuMapper.getSkuCount(shopId, goods.getId());
                if (skuCountVo != null) {
                    goods.setSelloutCount(skuCountVo.getSellout());
                    goods.setSkuCount(skuCountVo.getSellout() + skuCountVo.getSellin());
                }

                ShopRecommend shopRecommend = new ShopRecommend();
                shopRecommend.setShopId(String.valueOf(shopId));
                shopRecommend.setGoodsId(String.valueOf(goods.getId()));
                shopRecommend.setType("1");
                List<ShopRecommend> shopRecommends = shopRecommendMapper.selectShopRecommendList(shopRecommend);
                if (CollectionUtils.isEmpty(shopRecommends)) {
                    goods.setRecommend("0");
                } else {
                    goods.setRecommend("1");
                }
                shopRecommend.setType("2");
                shopRecommends = shopRecommendMapper.selectShopRecommendList(shopRecommend);
                if (CollectionUtils.isEmpty(shopRecommends)) {
                    goods.setPreferential("0");
                } else {
                    goods.setPreferential("1");
                }
                if (!StringUtils.isEmpty(goods.getRecommendTag())) {
                    goods.setRecommendTagList(JSON.parseArray(goods.getRecommendTag(), String.class));
                }
                goods.setSalesVolume(orderMapper.selectSalesVolume(shopId, goods.getId()));

                GoodsTag goodsTag = new GoodsTag();
                goodsTag.setState("1");
                goodsTag.setGoodsId(goods.getId());
                List<GoodsTag> goodsTags = goodsTagMapper.selectGoodsTagList(goodsTag);
                List<String> tags = goodsTags.stream().map(GoodsTag::getTag).collect(Collectors.toList());
                goods.setGoodsTagList(tags);
            }
            category.setGoodsList(goodsList);
        }
        return shopCategories;
    }

    @Override
    public List<GoodsSku> getSkuByGoodsId(long shopId, long goodsId) {
        return shopGoodsSkuMapper.selectSkuByShopAndGoods(shopId, goodsId);

    }

    @Override
    public String turn(Long shopId, Long goodsId, String pointer) {
        Goods goods = goodsMapper.selectGoodsById(goodsId);
        ShopGoods shopGoods = new ShopGoods();
        shopGoods.setShopId(shopId);
        List<ShopGoods> shopGoodsList = shopGoodsMapper.selectShopGoodsList(shopGoods);
        shopGoodsList = shopGoodsList.stream().filter(e -> {
            Goods shopGood = goodsMapper.selectGoodsById(e.getGoodsId());
            if (shopGood == null) {
                return false;
            }

            return ArrayUtil.hasCommonElement(shopGood.getCategory().split(","),
                    goods.getCategory().split(","));
        }).collect(Collectors.toList());

        int size = shopGoodsList.size();
        Long lastGoodsId = null;
        for (int i = 0; i < size; i++) {
            ShopGoods shopGood = shopGoodsList.get(i);
            if (goodsId.equals(shopGood.getGoodsId())) {
                if ("1".equals(pointer)) {
                    int index = i - 1;
                    if (index < 0) {
                        lastGoodsId = shopGood.getGoodsId();
                    } else {
                        lastGoodsId = shopGoodsList.get(index).getGoodsId();
                    }
                } else if ("2".equals(pointer)) {
                    int index = i + 1;
                    if (index >= size) {
                        lastGoodsId = shopGood.getGoodsId();
                    } else {
                        lastGoodsId = shopGoodsList.get(index).getGoodsId();
                    }
                }
            }
        }
        shopGoods.setGoodsId(goodsId);
        ShopGoods fristShopGoods = shopGoodsMapper.selectShopGoodsList(shopGoods).get(0);
        String turn = fristShopGoods.getTurn();
        shopGoods.setGoodsId(lastGoodsId);
        ShopGoods lastShopGoods = shopGoodsMapper.selectShopGoodsList(shopGoods).get(0);
        fristShopGoods.setTurn(lastShopGoods.getTurn());
        lastShopGoods.setTurn(turn);
        shopGoodsMapper.updateShopGoods(fristShopGoods);
        shopGoodsMapper.updateShopGoods(lastShopGoods);
        Machine machine = new Machine();
        machine.setShopId(shopId);
        List<Machine> machines = machineMapper.selectMachineList(machine);
        if (!machines.isEmpty()) {
            Machine machine1 = machines.get(0);
            updateApplication(Integer.parseInt(machine1.getId()));
        }
        return null;
    }

    @Override
    public List<String> getShopCity() {
        List<Shop> shops = shopMapper.selectShopList(new Shop());
        List<String> collect = shops.stream().map(Shop::getCity).collect(Collectors.toList());
        return collect;
    }

    @Override
    public Map<String, Map<String, Map<String, List<Shop>>>> getShopAreaTree() {
        List<Shop> shops = shopMapper.selectAllShop();
        return shops.stream().collect(Collectors.groupingBy(Shop::getProvince, LinkedHashMap::new,
                Collectors.groupingBy(Shop::getCity, LinkedHashMap::new,
                        Collectors.groupingBy(Shop::getZone, LinkedHashMap::new,
                                Collectors.mapping(shop -> new Shop(shop.getId(), shop.getCode(), shop.getName()), Collectors.toList())))));
    }

    @Override
    public String offGoods(JSONObject body) {
        Long shopId = body.getLong("shopId");
        String string = body.getString("goodsIds");
        List<String> goodsIds = JSONObject.parseArray(string, String.class);
        shopGoodsSkuMapper.deleteByGoodsIds(shopId, goodsIds);
        shopGoodsMapper.deleteByShopIdGoods(shopId, goodsIds);
        shopRecommendMapper.deleteByShopIdGoods(shopId, goodsIds);
        Machine machine = new Machine();
        machine.setShopId(shopId);
        List<Machine> machines = machineMapper.selectMachineList(machine);
        if (!machines.isEmpty()) {
            Machine machine1 = machines.get(0);
            updateApplication(Integer.parseInt(machine1.getId()));
        }
        return null;
    }

    @Override
    public List<Goods> selectGoodsList(String goodsName, String category, Long shopId) {
        List<Goods> goodsList = shopGoodsMapper.selectShopNoAddGoods(goodsName, category, shopId);
        for (Goods goods1 : goodsList) {
            GoodsCategory goodsCategory = goodsCategoryMapper.selectGoodsCategoryById(goods1.getCategory());
            if (goodsCategory != null) {
                goods1.setCategoryName(goodsCategory.getName());
            }
            int count = orderMapper.selectSalesVolume(null, goods1.getId());
            goods1.setSalesVolume(count);
        }
        return goodsList;

    }


    public AjaxResult checkSku(Long shopId, Long skuId, String machineCode) {
        ShopGoodsSku shopGoodsSku = new ShopGoodsSku();
        shopGoodsSku.setSkuId(skuId);
        if (shopId != null) {
            shopGoodsSku.setShopId(shopId);
        }
        if (StringUtils.isNotEmpty(machineCode)) {
            Machine machine = new Machine();
            machine.setCode(machineCode);
            List<Machine> machines = machineMapper.selectMachineList(machine);
            if (machines == null) {
                return AjaxResult.error("机器不存在");
            }
            Machine machine1 = machines.get(0);
            if (machine1.getShopId() == null) {
                return AjaxResult.error("该机器暂未绑定店铺");
            }
            shopGoodsSku.setShopId(machine1.getShopId());
        }
        shopGoodsSku.setIsDeleted("0");
        List<ShopGoodsSku> shopGoodsSkus = shopGoodsSkuMapper.selectShopGoodsSkuList(shopGoodsSku);
        if (shopGoodsSkus == null || shopGoodsSkus.isEmpty()) {
            return AjaxResult.error("该产品已经不存在");
        }
        ShopGoodsSku shopGoodsSku1 = shopGoodsSkus.get(0);
        JSONObject jsonObject = new JSONObject();
        jsonObject.put("state", shopGoodsSku1.getState());
        Goods goods = goodsMapper.selectGoodsById(shopGoodsSku1.getGoodsId());
        jsonObject.put("goods", getGoodsVo(goods, shopGoodsSku1.getShopId(), skuId));
        return AjaxResult.success(jsonObject);
    }

    public GoodsVo getGoodsVo(Goods good, Long shopId, Long skuId) {
        GoodsVo goodsVo = new GoodsVo();
        goodsVo.setGoodsId(String.valueOf(good.getId()));
        goodsVo.setName(good.getName());
        goodsVo.setDesc(good.getDesc());
        goodsVo.setDiscount(good.getDiscount().stripTrailingZeros().toPlainString());
        goodsVo.setPrice(good.getPrice().stripTrailingZeros().toPlainString());
        goodsVo.setPics(JSONObject.parseObject(good.getPics()));
        goodsVo.setRemarks(good.getRemarks());
        goodsVo.setSpecs(buildGoodsSpec(good.getSpec()));
        goodsVo.setSpecString(null);

        GoodsTag goodsTag = new GoodsTag();
        goodsTag.setState("1");
        goodsTag.setGoodsId(Long.parseLong(goodsVo.getGoodsId()));
        List<GoodsTag> goodsTags = goodsTagMapper.selectGoodsTagList(goodsTag);
        List<String> tags = goodsTags.stream().map(GoodsTag::getTag).collect(Collectors.toList());
        goodsVo.setTags(tags);
        //获取产品信息
        List<SkuVo> skuVos = shopGoodsSkuMapper.selectSkuByGoodAndShop(shopId, goodsVo.getGoodsId());
        buildSkuRules(skuVos, skuId);
        goodsVo.setSkus(skuVos);
        return goodsVo;
    }

    private JSONArray buildGoodsSpec(String specs) {
        List<Spec> specList = JSONObject.parseArray(specs, Spec.class);
        List<SpecVo> specVos = new ArrayList<>();
        for (Spec spec : specList) {
            SpecVo specVo = new SpecVo();
            specVos.add(specVo);
            specVo.setSpecId(String.valueOf(spec.getId()));
            specVo.setSpecName(spec.getName());
            List<SpecRuleVo> specRuleVos = new ArrayList<>();
            specVo.setRules(specRuleVos);
            for (SpecRule specRule : spec.getSpecRules()) {
                SpecRuleVo specRuleVo = new SpecRuleVo();
                specRuleVos.add(specRuleVo);
                specRuleVo.setRuleId(String.valueOf(specRule.getId()));
                specRuleVo.setPrice(specRule.getAmount().stripTrailingZeros().toPlainString());
                specRuleVo.setRuleName(specRule.getName());
                specRuleVo.setIsDefault(String.valueOf(specRule.getIsDefault()));
                specRuleVo.setIsRecommend(specRule.getIsRecommend());
            }

        }
        return JSONArray.parseArray(JSONObject.toJSONString(specVos));

    }

    private void buildSkuRules(List<SkuVo> skuVos, Long skuId) {
        if (CollectionUtils.isEmpty(skuVos)) {
            return;
        }
        for (SkuVo skuVo : skuVos) {
            if (Objects.equals(skuId, skuVo.getSkuId()) && !Objects.equals(skuVo.getIsDelete(), SkuDeleteState.NORMAL.getState())) {
                throw new ServiceException("规格已下架");
            }
            String rules = skuVo.getRulesString();
            List<SpecRule> specRuleList = JSONObject.parseArray(rules, SpecRule.class);
            boolean boo = true;
            List<SkuRuleVo> skuRuleVos = new ArrayList<>();
            for (SpecRule specRule : specRuleList) {
                SkuRuleVo skuRuleVo = new SkuRuleVo();
                skuRuleVos.add(skuRuleVo);
                skuRuleVo.setPrice(specRule.getAmount().stripTrailingZeros().toPlainString());
                skuRuleVo.setRuleId(String.valueOf(specRule.getId()));
                skuRuleVo.setSpecId(String.valueOf(specRule.getSpecId()));
                skuRuleVo.setRuleName(specRule.getName());
                if (0L == specRule.getIsDefault()) {
                    boo = false;
                }
            }
            skuVo.setIsDefault("0");
            if (boo) {
                skuVo.setIsDefault("1");
            }
            skuVo.setRules(JSONObject.parseArray(JSONObject.toJSONString(skuRuleVos)));
            skuVo.setRulesString(null);
        }
    }
}
