package com.soss.system.service.impl;

import com.soss.common.exception.ServiceException;
import com.soss.common.utils.DateUtils;
import com.soss.common.utils.StringUtils;
import com.soss.system.constants.OrderStatusConstant;
import com.soss.system.domain.Customer;
import com.soss.system.domain.Order;
import com.soss.system.domain.vo.customer.CustomerQueryVo;
import com.soss.system.domain.vo.customer.CustomerResultVo;
import com.soss.system.mapper.CustomerMapper;
import com.soss.system.mapper.OrderMapper;
import com.soss.system.mapper.ShopMapper;
import com.soss.system.service.ICustomerService;
import org.apache.commons.lang3.BooleanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.time.LocalDateTime;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Objects;

/**
 * 用户信息Service业务层处理
 * 
 * @author zxq
 * @date 2022-05-01
 */
@Service
public class CustomerServiceImpl implements ICustomerService 
{
    @Autowired
    private CustomerMapper customerMapper;
    @Autowired
    private OrderMapper orderMapper;
    @Autowired
    private ShopMapper shopMapper;

    @Override
    public Customer selectCustById(String custId) {
        return customerMapper.selectCustomerById(custId);
    }

    /**
     * 查询用户信息
     * 
     * @param id 用户信息主键
     * @param status
     * @return 用户信息
     */
    @Override
    public List<Order> selectCustomerById(String id, String status)
    {
        Order order = new Order();
        order.setUserId(id);
        List<String> statusList;
        if("1".equals(status)){
            statusList = Arrays.asList(OrderStatusConstant.PaidCancel, OrderStatusConstant.Unpaid, OrderStatusConstant.Paid, OrderStatusConstant.production, OrderStatusConstant.productionCompleted, OrderStatusConstant.Taking, OrderStatusConstant.completed);
        }else if("2".equals(status)){
            statusList = Arrays.asList(OrderStatusConstant.PaidCancel, OrderStatusConstant.UnpaidCancel, OrderStatusConstant.productionCancel, OrderStatusConstant.productionCompletedCancel);
        }else{
            statusList = Arrays.asList(OrderStatusConstant.refunding, OrderStatusConstant.refundFailed, OrderStatusConstant.PartialRefund, OrderStatusConstant.refund);
        }
        List<Order> orders = orderMapper.selectOrderByUserId(id, statusList);
        for (Order order1 : orders) {
            order1.setShop(shopMapper.selectShopById(order1.getShopId()));
        }
        return orders;
    }

    /**
     * 查询用户信息列表
     * 
     * @param customer 用户信息
     * @return 用户信息
     */
    @Override
    public List<CustomerResultVo> selectCustomerList(CustomerQueryVo customer)
    {

        List<CustomerResultVo> customerResultVos = customerMapper.selectCustomerByQuery(customer);
        return customerResultVos;
    }

    /**
     * 新增用户信息
     * 
     * @param customer 用户信息
     * @return 结果
     */
    @Override
    public int insertCustomer(Customer customer)
    {
        customer.setCreateTime(DateUtils.getNowDate());
        return customerMapper.insertCustomer(customer);
    }

    /**
     * 修改用户信息
     *
     * @param customer 用户信息
     * @return 结果
     */
    @Override
    @Transactional
    public int updateCustomer(Customer customer) {
        Customer customerExist = this.selectCustById(customer.getId());
        if (customerExist == null) {
            throw new ServiceException("用户不存在");
        }
        // 用户修改昵称时，修改订单冗余字段-用户昵称
        if (StringUtils.isNotEmpty(customer.getUserName()) && !Objects.equals(customerExist.getUserName(), customer.getUserName())) {
            orderMapper.updateCustName(customer.getUserName(), customer.getId());
        }
        return customerMapper.updateCustomer(customer);
    }

    /**
     * 修改用户授权信息
     */
    @Override
    public int updateSendMsgFlag(Customer customer) {
        if (!BooleanUtils.isTrue(customer.getRememberFlag())) {
            LocalDateTime date = DateUtils.addDaysAndGetEnd(LocalDateTime.now(), 6);
            customer.setFlagExpireDate(date);
        }
        return customerMapper.updateSendMsgFlag(customer);
    }

    @Override
    public void updateCustBirthday(Customer customer) {
        Customer cust = customerMapper.selectCustomerById(customer.getId());
        if (cust.getBirthday() != null) {
            throw new ServiceException("生日只能修改一次");
        }
        customerMapper.updateCustomer(customer);
    }

    /**
     * 批量删除用户信息
     *
     * @param ids 需要删除的用户信息主键
     * @return 结果
     */
    @Override
    public int deleteCustomerByIds(Long[] ids) {
        return customerMapper.deleteCustomerByIds(ids);
    }

    /**
     * 删除用户信息信息
     * 
     * @param id 用户信息主键
     * @return 结果
     */
    @Override
    public int deleteCustomerById(Long id)
    {
        return customerMapper.deleteCustomerById(id);
    }

    @Override
    public Map<String, Object> getCustSimpleOrderStat(String custId) {
        return orderMapper.getCustOrderSimpleStat(custId);
    }

    @Override
    public int insertCustFaceInfo(String openId, String faceInfoId){
        return customerMapper.insertCustFaceId(openId, faceInfoId);
    }
}
