package com.soss.system.service.impl;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.soss.common.core.domain.AjaxResult;
import com.soss.common.core.domain.model.LoginUser;
import com.soss.common.enums.CouponUserState;
import com.soss.common.enums.ShopState;
import com.soss.common.exception.ServiceException;
import com.soss.common.utils.DateUtils;
import com.soss.common.utils.StringUtils;
import com.soss.common.utils.spring.SpringUtils;
import com.soss.system.constants.OrderStatusConstant;
import com.soss.system.domain.*;
import com.soss.system.domain.vo.OrderQuery;
import com.soss.system.domain.vo.orderTaking.CategoryVo;
import com.soss.system.domain.vo.orderTaking.GoodsVo;
import com.soss.system.domain.vo.orderTaking.OrderTakingVo;
import com.soss.system.domain.vo.orderTaking.SkuVo;
import com.soss.system.mapper.*;
import com.soss.system.service.IOrderService;
import com.soss.system.utils.DistanceUtil;
import com.soss.system.utils.SendSmsUtil;
import com.soss.system.weixin.util.SendMessageUtils;
import org.apache.commons.lang3.BooleanUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.data.redis.support.atomic.RedisAtomicInteger;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.text.SimpleDateFormat;
import java.time.Duration;
import java.time.LocalDateTime;
import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 订单Service业务层处理
 *
 * @author zxq
 * @date 2022-04-28
 */
@Service
public class OrderServiceImpl implements IOrderService {
    @Autowired
    private OrderMapper orderMapper;
    @Autowired
    private ShopGoodsSkuMapper shopGoodsSkuMapper;
    @Autowired
    private OrderDetailMapper orderDetailMapper;
    @Autowired
    private GoodsMapper goodsMapper;
    @Autowired
    private MachineMapper machineMapper;
    @Autowired
    private MachineApiServiceImpl machineApiService;
    @Autowired
    private ShopMapper shopMapper;
    @Autowired
    private CouponUserMapper couponUserMapper;
    @Autowired
    private OrderOperationLogServiceImpl orderOperationLogService;
    @Autowired
    private OrderSnapshotServiceImpl orderSnapshotService;
    @Autowired
    private WechatMessageServiceImpl wechatMessageService;
    @Autowired
    private SendMessageUtils sendMessageUtils;
    @Autowired
    private OrderTakingServiceImpl orderTakingService;
    @Autowired
    private SysConfigServiceImpl sysConfigService;

    @Autowired
    SendSmsUtil sendSmsUtil;

    /**
     * 查询订单
     *
     * @param id 订单主键
     * @return 订单
     */
    @Override
    public Order selectOrderById(Long id) {
        Order order = orderMapper.selectOrderById(id);
        order.setShop(shopMapper.selectShopById(order.getShopId()));
        OrderSnapshot orderSnapshot = orderSnapshotService.selectOrderSnapshotByOrderId(order.getId());
        String snapshot = orderSnapshot.getSnapshot();
        List<OrderDetail> orderDetails = JSONObject.parseArray(snapshot, OrderDetail.class);
        List<Long> detailIds = orderDetails.stream().map(OrderDetail::getId).collect(Collectors.toList());
        List<OrderDetail> detailsActual = orderDetailMapper.selectOrderDetailsByIds(detailIds);
        Map<Long, Long> refundIdMap = detailsActual.stream().filter(da -> da.getRefundId() != null).collect(Collectors.toMap(OrderDetail::getId, OrderDetail::getRefundId));
        orderDetails.forEach(od -> od.setRefundId(refundIdMap.get(od.getId())));
        order.setOrderDetails(orderDetails);
        return order;
    }

    public Order selectById(Long id) {
        return orderMapper.selectOrderById(id);
    }

    public void updateById(Order order) {
        orderMapper.updateOrder(order);
    }

    /**
     * 查询订单列表
     *
     * @param order 订单
     * @return 订单
     */
    @Override
    public List<Order> selectOrderList(OrderQuery order) {
        if (StringUtils.isNotEmpty(order.getCreateAtEnd())) {
            order.setCreateAtEnd(order.getCreateAtEnd() + " 23:59:59");
        }
        if (StringUtils.isNotEmpty(order.getPayAtEnd())) {
            order.setPayAtEnd(order.getPayAtEnd() + " 23:59:59");
        }
        return orderMapper.selectOrderByQuery(order);
    }

    public List<Order> selectOrderList(Order order) {
        List<Order> orders = orderMapper.selectOrderList(order);
        return orders;
    }

    /**
     * 新增订单
     *
     * @param loginUser
     * @param order        订单
     * @param couponUserId
     * @return 结果
     */
    @Override
    @Transactional
    public AjaxResult insertOrder(LoginUser loginUser, Order order, Integer couponUserId) {
        order.setState(OrderStatusConstant.Unpaid);
        order.setUserId(loginUser.getOpenId());
        order.setUserName(loginUser.getUsername());
        order.setUserPhone(loginUser.getUser().getPhonenumber());
        order.setCreatedAt(new Date());
        order.setOrderNum(getOrderNum(order.getShopId()));
        order.setOrderNo(getOrderNo(order.getSource()));
        order.setUpdatedAt(new Date());
        orderMapper.insertOrder(order);

        this.orderDetailSplit(order);
        orderDetailMapper.insertBatch(order.getOrderDetails());

        // 更新用户的优惠券状态
        if (couponUserId != null) {
            CouponUser couponUser = new CouponUser(couponUserId, order.getId(), order.getOrderNo(), CouponUserState.USED.getState());
            couponUserMapper.updateCouponUser(couponUser);
        }

        orderOperationLogService.insertOrderOperationLog(order);
        orderSnapshotService.saveOrderSnapshot(order);
        //发送数据给机器
        machineApiService.createOrder(order);
        return AjaxResult.success(order.getOrderNo());
    }

    private void orderDetailSplit(Order order) {
        List<OrderDetail> orderDetails = order.getOrderDetails();
        List<OrderDetail> detailsSave = new LinkedList<>();
        for (OrderDetail orderDetail : orderDetails) {
            orderDetail.setOrderId(order.getId());
            orderDetail.setCreatedAt(new Date());
            orderDetail.setUpdatedAt(new Date());
            orderDetail.setMachineId(order.getMachineId());

            BigDecimal amountShould = orderDetail.getAmountShould();
            BigDecimal couponAmount = orderDetail.getCouponAmount();
            if (couponAmount == null) {
                couponAmount = BigDecimal.ZERO;
            }
            BigDecimal realAmount = orderDetail.getRealAmount();
            int skuNum = orderDetail.getNum();

            BigDecimal amountShouldUnit = amountShould.divide(new BigDecimal(skuNum), 2, BigDecimal.ROUND_HALF_UP);
            BigDecimal couponAmountUnit = couponAmount.divide(new BigDecimal(skuNum), 2, BigDecimal.ROUND_HALF_UP);
            BigDecimal realAmountUnit = realAmount.divide(new BigDecimal(skuNum), 2, BigDecimal.ROUND_HALF_UP);

            for (int i = 1; i <= skuNum; i++) {
                OrderDetail orderDetailLst = new OrderDetail();
                BeanUtils.copyProperties(orderDetail, orderDetailLst);
                orderDetailLst.setNum(1);
                if (i == skuNum) {
                    orderDetailLst.setAmountShould(amountShould.subtract(amountShouldUnit.multiply(new BigDecimal(i - 1))));
                    orderDetailLst.setCouponAmount(couponAmount.subtract(couponAmountUnit.multiply(new BigDecimal(i - 1))));
                    orderDetailLst.setRealAmount(realAmount.subtract(realAmountUnit.multiply(new BigDecimal(i - 1))));
                } else {
                    orderDetailLst.setAmountShould(amountShouldUnit);
                    orderDetailLst.setCouponAmount(couponAmountUnit);
                    orderDetailLst.setRealAmount(realAmountUnit);
                }
                detailsSave.add(orderDetailLst);
            }
        }
        order.setOrderDetails(detailsSave);
    }

    @Override
    public void checkShopInfo(Order order) {
        Shop shop = shopMapper.selectShopById(order.getShopId());
        String startTime = shop.getStartTime();
        String endTime = shop.getEndTime();
        String[] split = startTime.split(":");
        String[] split1 = endTime.split(":");
        Calendar startBusiness = Calendar.getInstance();
        startBusiness.set(Calendar.HOUR_OF_DAY, Integer.parseInt(split[0]));
        startBusiness.set(Calendar.MINUTE, Integer.parseInt(split[1]));
        Calendar endBusiness = Calendar.getInstance();
        endBusiness.set(Calendar.HOUR_OF_DAY, Integer.parseInt(split1[0]));
        endBusiness.set(Calendar.MINUTE, Integer.parseInt(split1[1]));
        long l = System.currentTimeMillis();
        if (startBusiness.getTimeInMillis() > l || endBusiness.getTimeInMillis() < l) {
            throw new ServiceException("当前时间本店休息中，暂停点单~");
        }
        if (!Objects.equals(ShopState.OPEN.getState(), shop.getState())) {
            throw new ServiceException("店铺非营业状态不能下单");
        }
        if (shop.getDistanceLimit() != null && shop.getDistanceLimit() > 0) {
            double distance = DistanceUtil.getRealDistance(order.getLng().doubleValue(), order.getLat().doubleValue(), Double.parseDouble(shop.getLng()), Double.parseDouble(shop.getLat()));
            if (distance > shop.getDistanceLimit() * 1000) {
                throw new ServiceException("超过店铺下单的距离范围");
            }
        }
    }

    /**
     * 订单重写从这里开始，这里要承担大部分的业务逻辑，逻辑集中化处理
     */
    @Override
    public int updateOrder(Order order) {
        String state = order.getState();
        switch (state) {
            case OrderStatusConstant.Paid:
                paidState(order);
                break;
            case OrderStatusConstant.production:
                productionState(order);
                break;
            case OrderStatusConstant.productionCompleted:
                productionCompletedState(order);
                break;
            case OrderStatusConstant.Taking:
                takingState(order);
                break;
            case OrderStatusConstant.completed:
                completedState(order);
                break;
            case OrderStatusConstant.timeout:
                timeoutState(order);
                break;
            case OrderStatusConstant.UnpaidCancel:
            case OrderStatusConstant.PaidCancel:
                unpaidCancelState(order);
                break;
            case OrderStatusConstant.refunding:
                refundingState(order);
                break;
            case OrderStatusConstant.PartialRefund:
            case OrderStatusConstant.refund:
                partialRefundState(order);
                break;
        }

        order.setUpdatedAt(new Date());
        return orderMapper.updateOrder(order);
    }

    private void timeoutState(Order order) {
//        jiGuangPushService.pushOrderState(order);
        if (BooleanUtils.isTrue(order.getSendMsgFlag2())) {
            sendMessageUtils.sendWxMsg(order);
        }
    }

    /**
     * 部分退款
     *
     * @param order
     */
    private void partialRefundState(Order order) {
//        jiGuangPushService.pushOrderState(order);
        machineApiService.updateOrder(order);
    }

    /**
     * 制作完成 机器给我们
     * 记录日志
     * 极光推送
     *
     * @param order
     */
    private void completedState(Order order) {
        order.setFinishTime(new Date());
        orderOperationLogService.insertOrderOperationLog(order);
//        jiGuangPushService.pushOrderState(order);
        if (BooleanUtils.isTrue(order.getSendMsgFlag3())) {
            sendMessageUtils.sendWxMsg(order);
        }
    }

    /**
     * 退款中
     * 极光推送
     * 推送给机器
     *
     * @param order
     */
    private void refundingState(Order order) {
//        jiGuangPushService.pushOrderState(order);
        machineApiService.updateOrder(order);
    }

    /**
     * 未支付取消  支付取消
     * 记录日志
     * 发送给机器
     * 极光推送
     *
     * @param order
     */
    private void unpaidCancelState(Order order) {
        orderOperationLogService.insertOrderOperationLog(order);
        machineApiService.updateOrder(order);
//        jiGuangPushService.pushOrderState(order);

    }

    /**
     * 取餐中
     * 极光推送
     *
     * @param order
     */
    private void takingState(Order order) {
        orderOperationLogService.insertOrderOperationLog(order);
//        jiGuangPushService.pushOrderState(order);
    }

    /**
     * 制作完成待取餐
     * 记录日志
     * 极光推送
     * 消息记录
     * 发送推送消息
     *
     * @param order
     */
    private void productionCompletedState(Order order) {
        orderOperationLogService.insertOrderOperationLog(order);
//        jiGuangPushService.pushOrderState(order);
        wechatMessageService.insertWechatMessage(order.getUserId(), order.getId(), "制作完成,您的订单已经制作完成，现在可以去取餐啦！");
        if (BooleanUtils.isTrue(order.getSendMsgFlag2())) {
            sendMessageUtils.sendWxMsg(order);
            sendSmsUtil.sendSms(order.getUserPhone(), "SMS_254715682", null);
        }
    }

    /**
     * 制作中，机器推送给我们
     * 记录日志
     * 发送消息给安卓
     * 消息记录
     * 发送推送消息
     *
     * @param order
     */
    private void productionState(Order order) {
        orderOperationLogService.insertOrderOperationLog(order);
//        jiGuangPushService.pushOrderState(order);
        wechatMessageService.insertWechatMessage(order.getUserId(), order.getId(), "制作提醒,您的订单已经开始制作，马上就能享受美味了！");
        if (BooleanUtils.isTrue(order.getSendMsgFlag2())) {
            sendMessageUtils.sendWxMsg(order);
        }
    }

    /**
     * 已付款要干这四件事
     * 记录日志，
     * 发送给机器
     * 发送给安卓
     * 消息记录
     * 发送推送消息
     *
     * @param order
     */
    private void paidState(Order order) {
        orderOperationLogService.insertOrderOperationLog(order);
        machineApiService.updateOrder(order);
        //极光推送
//        jiGuangPushService.pushOrderState(order);
        wechatMessageService.insertWechatMessage(order.getUserId(), order.getId(), "下单成功,您已经下单成功了~");
        if (BooleanUtils.isTrue(order.getSendMsgFlag1())) {
            sendMessageUtils.sendWxMsg(order);
        }
    }

    @Override
    public AjaxResult check(Order order) {
        //验证订单状态
        List<OrderDetail> orderDetails = order.getOrderDetails();
        if (orderDetails == null || orderDetails.isEmpty()) {
            return AjaxResult.error("订单信息不能为空");
        }

        String orderLimit = sysConfigService.selectConfigByKey("goods.limit");
        int countOfOrder = StringUtils.isEmpty(orderLimit) ? 9 : Integer.parseInt(orderLimit);
        if (orderDetails.size() > countOfOrder) {
            return AjaxResult.error(String.format("单次最多购买%s杯，请结算后再次下单", countOfOrder));
        }

        for (OrderDetail orderDetail : orderDetails) {
            ShopGoodsSku shopGoodsSku = new ShopGoodsSku();
            shopGoodsSku.setSkuId(orderDetail.getSkuId());
            shopGoodsSku.setShopId(order.getShopId());
            shopGoodsSku.setIsDeleted("0");
            List<ShopGoodsSku> shopGoodsSkus = shopGoodsSkuMapper.selectShopGoodsSkuList(shopGoodsSku);
            if (shopGoodsSkus == null || shopGoodsSkus.isEmpty()) {
                return AjaxResult.error("本商品已不存在：" + orderDetail.getGoodsName());
            }
            ShopGoodsSku shopGoodsSku1 = shopGoodsSkus.get(0);
            if (!"1".equals(shopGoodsSku1.getState())) {
                return AjaxResult.error("本商品已售罄或已经下架：" + orderDetail.getGoodsName());
            }
        }
        return AjaxResult.success();
    }

    @Override
    public List<Order> getMyOrder(String openId) {
        return orderMapper.selectByUserId(openId);
    }

    @Override
    public void wrapperOrder(List<Order> orders, boolean snapshotFlag) {
        if (CollectionUtils.isEmpty(orders)) {
            return;
        }

        List<Long> shopIds = orders.stream().map(Order::getShopId).distinct().collect(Collectors.toList());
        List<Shop> shops = shopMapper.selectShopByIds(shopIds);
        Map<Long, Shop> shopMap = shops.stream().collect(Collectors.toMap(Shop::getId, Function.identity()));
        Map<Long, String> snapshotMap = null;
        if (snapshotFlag) {
            List<Long> orderIds = orders.stream().map(Order::getId).collect(Collectors.toList());
            List<OrderSnapshot> snapshots = orderSnapshotService.selectOrderSnapshotByOrderIds(orderIds);
            snapshotMap = snapshots.stream().collect(Collectors.toMap(OrderSnapshot::getOrderId, OrderSnapshot::getSnapshot));
        }
        for (Order order : orders) {
            order.setShop(shopMap.get(order.getShopId()));
            if (snapshotFlag) {
                order.setOrderDetails(JSONArray.parseArray(snapshotMap.get(order.getId()), OrderDetail.class));
            }
        }
    }

    @Override
    public void wrapperOrderDetailForApplet(Order order) {
        Map<Long, List<OrderDetail>> orderDetailSkuMap = order.getOrderDetails().stream()
                .collect(Collectors.groupingBy(OrderDetail::getSkuId, Collectors.toList()));
        List<OrderDetail> orderDetails = new ArrayList<>();
        for (List<OrderDetail> mapVal : orderDetailSkuMap.values()) {
            OrderDetail detail = mapVal.get(0);
            if (mapVal.size() > 1) {
                detail.setAmountShould(mapVal.stream().map(OrderDetail::getAmountShould).reduce(BigDecimal.ZERO, BigDecimal::add));
                detail.setCouponAmount(mapVal.stream().map(OrderDetail::getCouponAmount).reduce(BigDecimal.ZERO, BigDecimal::add));
                detail.setRealAmount(mapVal.stream().map(OrderDetail::getRealAmount).reduce(BigDecimal.ZERO, BigDecimal::add));
                detail.setNum(mapVal.size());
            }
            orderDetails.add(detail);
        }
        order.setOrderDetails(orderDetails);
    }

    @Override
    @Transactional
    public int cancel(Long orderId) {
        Order order = orderMapper.selectOrderById(orderId);
        if (OrderStatusConstant.Unpaid.equals(order.getState())) {
            order.setState("8");
            order.setUpdatedAt(new Date());
            this.updateOrder(order);

        } else if (OrderStatusConstant.Paid.equals(order.getState())) {
            order.setState("9");
            order.setUpdatedAt(new Date());
            this.updateOrder(order);
        } else {
            throw new ServiceException("该订单不允许取消");
        }

        return 0;
    }


    /**
     * 获取机器顺序号
     *
     * @param shopId
     * @return
     */
    private String getOrderNum(Long shopId) {
        Calendar calendar = Calendar.getInstance();
//        String key = calendar.getTime().toString().substring(0, 3);
        String key = String.valueOf(calendar.get(Calendar.DAY_OF_WEEK) - 1);
        SimpleDateFormat simpleDateFormat = new SimpleDateFormat("yyyy-MM-dd");
        String format = simpleDateFormat.format(calendar.getTime());
        String redisKey = "shopId" + format + shopId;
        RedisAtomicInteger redisAtomicInteger = new RedisAtomicInteger(redisKey, SpringUtils.getBean(StringRedisTemplate.class).getConnectionFactory());
        if (redisAtomicInteger.get() == 0) {
            redisAtomicInteger.set((int) (998 * Math.random() + 1));
            LocalDateTime now = LocalDateTime.now();
            LocalDateTime expireTime = DateUtils.addDaysAndGetBegin(now, 1);
            long seconds = Duration.between(now, expireTime).getSeconds();
            redisAtomicInteger.expire(seconds, TimeUnit.SECONDS);
        }
        int delta = (int) (Math.random() * 8 + 1);
        int orderNum = redisAtomicInteger.addAndGet(delta);
        if (orderNum > 999) {
            redisAtomicInteger.set(1);
            orderNum = 1;
        }
        return key + String.format("%03d", orderNum);
    }

    //获取订单号
    public String getOrderNo(Integer source) {
        String channel = "M";
        if (source != null) {
            if (source == 1) {
                channel = "A";
            } else if (source == 2) {
                channel = "B";
            }
        }
        SimpleDateFormat simpleDateFormat = new SimpleDateFormat("yyyyMMddHH");
        String format = simpleDateFormat.format(new Date());
        format = format.substring(2);
        String key = "orderNo";
        RedisAtomicInteger redisAtomicInteger = new RedisAtomicInteger(key, SpringUtils.getBean(StringRedisTemplate.class).getConnectionFactory());
        int andIncrement = redisAtomicInteger.getAndIncrement();
        if (andIncrement > 9999) {
            redisAtomicInteger.set(0);
            andIncrement = 0;
        }
        String format1 = String.format("%04d", andIncrement);
        int randomNumber = new Random().nextInt(99);
        return "N" + format + channel + format1 + randomNumber;
    }

    public List<JSONObject> getOrderInfo(String machineCode) {
        Machine machine = machineMapper.selectMachineByCode(machineCode);
        List<String> status = Arrays.asList("2", "3", "4");
        List<Order> orders = orderMapper.selectOrder(status, machine.getId());
        Map<String, List<Order>> collect = orders.stream().collect(Collectors.groupingBy(Order::getState));
        List<JSONObject> objects = new ArrayList<>();
        for (String key : collect.keySet()) {
            JSONObject jsonObject = new JSONObject();
            objects.add(jsonObject);
            jsonObject.put("state", key);
            List<Order> orders1 = collect.get(key);
            List<JSONObject> jsonObjects = new ArrayList<>();
            jsonObject.put("orders", jsonObjects);
            for (Order order : orders1) {
                JSONObject jsonObject1 = new JSONObject();
                jsonObjects.add(jsonObject1);
                jsonObject1.put("orderId", order.getId());
                jsonObject1.put("orderNumber", order.getOrderNum());
                jsonObject1.put("orderNo", order.getOrderNo());
            }
        }
        return objects;

    }


    public String refundOrder(Long orderId) {
        Order order = orderMapper.selectOrderById(orderId);
        List<String> status = Arrays.asList("2", "3");
        if (status.contains(order.getState())) {
            //todo 创建退款单，进行退款
        } else {
            //todo 创建退款单

        }
        return null;
    }

    public Map<String, String> getMyFristOrder(String openId, String shopId) {
        List<String> status = Arrays.asList("2", "3", "4");
        Order order = orderMapper.selectHomeByUserId(openId, status, shopId);
        if (order == null) {
            return null;
        }
        OrderDetail orderDetail = new OrderDetail();
        orderDetail.setOrderId(order.getId());
        List<OrderDetail> orderDetails = orderDetailMapper.selectOrderDetailList(orderDetail);
        order.setOrderDetails(orderDetails);
        String waitTime = getWaitTimeByOrderId(order.getId());
        Map<String, String> map = new HashMap<>();
        map.put("waitTime", waitTime);
        map.put("pickCode", order.getPickCode());
        map.put("orderNum", order.getOrderNum());
        map.put("state", order.getState());
        return map;
    }

    public int getWaitTime(Order order) {
        Long shopId = order.getShopId();
        List<String> status = Arrays.asList(OrderStatusConstant.Paid, OrderStatusConstant.production);
        List<Order> orders = orderMapper.selectOrderByShopId(status, shopId);
        List<OrderDetail> list = new ArrayList<>();
        list.addAll(order.getOrderDetails());
        for (Order order1 : orders) {
            OrderDetail orderDetail = new OrderDetail();
            orderDetail.setOrderId(order1.getId());
            List<OrderDetail> orderDetails = orderDetailMapper.selectOrderDetailList(orderDetail);
            if (!CollectionUtils.isEmpty(orderDetails)) {
                list.addAll(orderDetails);
            }
        }
        Long takeTimeCount = 0L;
        for (OrderDetail orderDetail : list) {
            Goods goods = goodsMapper.selectGoodsById(orderDetail.getGoodsId());
            if (goods == null) {
                continue;
            }
            Long takeTime = goods.getTakeTime();
            takeTime = orderDetail.getNum() * takeTime;
            takeTimeCount += takeTime;
        }

        return (int) (takeTimeCount / 60);
    }

    public String getWaitTimeByOrderId(Long orderId) {
        List<String> status = Arrays.asList(OrderStatusConstant.Paid, OrderStatusConstant.production);
        List<Order> orders = orderMapper.selectBeforeOrder(status, orderId);
        List<OrderDetail> list = new ArrayList<>();
        for (Order order1 : orders) {
            OrderDetail orderDetail = new OrderDetail();
            orderDetail.setOrderId(order1.getId());
            List<OrderDetail> orderDetails = orderDetailMapper.selectOrderDetailList(orderDetail);
            if (CollectionUtils.isEmpty(orderDetails)) {

            } else {
                list.addAll(orderDetails);
            }
        }
        Long takeTimeCount = 0L;
        for (OrderDetail orderDetail : list) {
            Goods goods = goodsMapper.selectGoodsById(orderDetail.getGoodsId());
            if (goods != null) {
                Long takeTime = goods.getTakeTime();
                takeTime = orderDetail.getNum() * takeTime;
                takeTimeCount += takeTime;
            }
        }

        return String.valueOf((int) (takeTimeCount / 60));
    }


    public List<GoodsVo> getNextOrder(Long orderId, Long shopId) {
        OrderDetail orderDetail = new OrderDetail();
        orderDetail.setOrderId(orderId);
        List<OrderDetail> orderDetails = orderDetailMapper.selectOrderDetailList(orderDetail);
        List<Long> skuIds = orderDetails.stream().map(OrderDetail::getSkuId).collect(Collectors.toList());
        OrderTakingVo infoByShop = orderTakingService.getInfoByShop(shopId);
        List<CategoryVo> categorys = infoByShop.getCategorys();
        List<GoodsVo> ccs = new ArrayList<>();
        if (categorys != null && !categorys.isEmpty()) {
            Set<String> cateAndGoodsIdSet = new HashSet<>();
            for (CategoryVo category : categorys) {
                if (category.getId() == 0) {
                    continue;
                }
                List<GoodsVo> goods = category.getGoods();
                if (goods == null || goods.isEmpty()) {
                    continue;
                }
                for (GoodsVo good : goods) {
                    String cateAndGoodsIdStr = good.getCategory() + good.getGoodsId();
                    if (cateAndGoodsIdSet.contains(cateAndGoodsIdStr)) {
                        continue;
                    } else {
                        cateAndGoodsIdSet.add(cateAndGoodsIdStr);
                    }
                    List<SkuVo> skuVoList = new ArrayList<>();
                    if (good.getSkus() == null || good.getSkus().isEmpty()) {
                        continue;
                    }
                    for (SkuVo skus : good.getSkus()) {
                        if (skuIds.contains(skus.getSkuId())) {
                            skuVoList.add(skus);
                        }
                    }
                    if (!skuVoList.isEmpty()) {
                        ccs.add(good);
                        good.setSkus(skuVoList);
                    }
                }

            }
        }
        return ccs;


    }
}
