package com.soss.system.service.impl;

import com.alibaba.fastjson.JSON;
import com.soss.common.exception.ServiceException;
import com.soss.system.domain.Machine;
import com.soss.system.domain.ShopGoods;
import com.soss.system.domain.Spec;
import com.soss.system.domain.SpecRule;
import com.soss.system.domain.vo.orderTaking.SkuVo;
import com.soss.system.mapper.*;
import com.soss.system.service.IShopGoodsService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * 店铺和商品关联Service业务层处理
 *
 * @author zxq
 * @date 2022-04-28
 */
@Service
public class ShopGoodsServiceImpl implements IShopGoodsService {
    @Autowired
    private ShopGoodsMapper shopGoodsMapper;
    @Autowired
    private ShopGoodsSkuMapper shopGoodsSkuMapper;
    @Autowired
    private GoodsMapper goodsMapper;
    @Autowired
    private SpecMapper specMapper;
    @Autowired
    private MachineMapper machineMapper;
    @Autowired
    private MachineApiServiceImpl machineApiService;

    /**
     * 查询店铺和商品关联
     *
     * @param id 店铺和商品关联主键
     * @return 店铺和商品关联
     */
    @Override
    public ShopGoods selectShopGoodsById(String id) {
        return shopGoodsMapper.selectShopGoodsById(id);
    }

    /**
     * 查询店铺和商品关联列表
     *
     * @param shopGoods 店铺和商品关联
     * @return 店铺和商品关联
     */
    @Override
    public List<ShopGoods> selectShopGoodsList(ShopGoods shopGoods) {
        return shopGoodsMapper.selectShopGoodsList(shopGoods);
    }

    /**
     * 新增店铺和商品关联
     *
     * @param shopGoods 店铺和商品关联
     * @return 结果
     */
    @Override
    public int insertShopGoods(ShopGoods shopGoods) {
        return shopGoodsMapper.insertShopGoods(shopGoods);
    }

    /**
     * 修改店铺和商品关联
     *
     * @param shopGoods 店铺和商品关联
     * @return 结果
     */
    @Override
    public int updateShopGoods(ShopGoods shopGoods) {
        return shopGoodsMapper.updateShopGoods(shopGoods);
    }

    /**
     * 批量删除店铺和商品关联
     *
     * @param ids 需要删除的店铺和商品关联主键
     * @return 结果
     */
    @Override
    public int deleteShopGoodsByIds(String[] ids) {
        return shopGoodsMapper.deleteShopGoodsByIds(ids);
    }

    /**
     * 删除店铺和商品关联信息
     *
     * @param id 店铺和商品关联主键
     * @return 结果
     */
    @Override
    public int deleteShopGoodsById(String id) {
        return shopGoodsMapper.deleteShopGoodsById(id);
    }

    @Override
    public List<Spec> getShopGoodsSpecs(Long shopId, Long goodsId) {
        List<SkuVo> skuVos = shopGoodsSkuMapper.selectSkuByGoodAndShop(shopId, goodsId.toString());
        if (CollectionUtils.isEmpty(skuVos)) {
            return null;
        }

        Map<String, List<SkuVo>> skuMap = skuVos.stream().collect(Collectors.groupingBy(SkuVo::getState));
        if (CollectionUtils.isEmpty(skuMap)) {
            return null;
        }

        Map<Long, SpecRule> selloutRules = new HashMap<>();
        Map<Long, SpecRule> sellinRules = new HashMap<>();

        List<SkuVo> selloutSkus = skuMap.get("2");
        if (!CollectionUtils.isEmpty(selloutSkus)) {
            selloutSkus.forEach(skuVo -> {
                List<SpecRule> rules = JSON.parseArray(skuVo.getRulesString(), SpecRule.class);
                if (!CollectionUtils.isEmpty(rules)) {
                    rules.forEach(rule -> {
                        rule.setState(2);
                        selloutRules.put(rule.getId(), rule);
                    });
                }
            });
        }

        List<SkuVo> sellinSkus = skuMap.get("1");
        if (!CollectionUtils.isEmpty(sellinSkus)) {
            sellinSkus.forEach(skuVo -> {
                List<SpecRule> rules = JSON.parseArray(skuVo.getRulesString(), SpecRule.class);
                if (!CollectionUtils.isEmpty(rules)) {
                    rules.forEach(rule -> {
                        rule.setState(1);
                        sellinRules.put(rule.getId(), rule);
                    });
                }
            });
        }

        // 从售罄sku选项列表中删除 未售罄sku选项
        sellinRules.values().forEach(rule -> selloutRules.remove(rule.getId()));

        Map<Long, List<SpecRule>> selloutRulesBySpec = selloutRules.values().stream().collect(Collectors.groupingBy(SpecRule::getSpecId));
        Map<Long, List<SpecRule>> sellinRulesBySpec = sellinRules.values().stream().collect(Collectors.groupingBy(SpecRule::getSpecId));

        List<Long> specIds = new ArrayList<>();
        specIds.addAll(sellinRules.values().stream().map(SpecRule::getSpecId).collect(Collectors.toList()));
        specIds.addAll(selloutRules.values().stream().map(SpecRule::getSpecId).collect(Collectors.toList()));

        // 构建返回数据
        List<Spec> specVos = specMapper.selectSpecByIds(specIds);
        if (CollectionUtils.isEmpty(specVos)) {
            return null;
        }
        specVos.forEach(spec -> {
            if (CollectionUtils.isEmpty(spec.getSpecRules())) {
                spec.setSpecRules(new ArrayList<>());
            }
            if (!CollectionUtils.isEmpty(sellinRulesBySpec.get(spec.getId()))) {
                spec.getSpecRules().addAll(sellinRulesBySpec.get(spec.getId()));
            }
            if (!CollectionUtils.isEmpty(selloutRulesBySpec.get(spec.getId()))) {
                spec.getSpecRules().addAll(selloutRulesBySpec.get(spec.getId()));
            }
        });

        return specVos;
    }

    public Boolean updateShopGoodsSpecState(Long shopId, Long goodsId, Long ruleId, Integer state) {
        if (shopId == null || shopId <= 0) {
            throw new ServiceException("店铺id错误");
        }
        if (goodsId == null || goodsId <= 0) {
            throw new ServiceException("商品id错误");
        }
        if (ruleId == null || ruleId <= 0) {
            throw new ServiceException("选项id错误");
        }
        if (state == null) {
            throw new ServiceException("状态不能为空");
        }
        if (state != 1 && state != 2) {
            throw new ServiceException("状态错误，1-2");
        }

        List<SkuVo> skuVos = shopGoodsSkuMapper.selectSkuByGoodAndShop(shopId, goodsId.toString());
        if (CollectionUtils.isEmpty(skuVos)) {
            return false;
        }

        // 本店铺本商品 所有规格，选项状态
        List<Spec> specs = getShopGoodsSpecs(shopId, goodsId);
        if (CollectionUtils.isEmpty(specs)) {
            return false;
        }

        // 包含本次修改选项的sku列表
        List<String> skuIds = new ArrayList<>();
        skuVos.forEach(skuVo -> {
            List<SpecRule> rules = JSON.parseArray(skuVo.getRulesString(), SpecRule.class);
            if (CollectionUtils.isEmpty(rules)) {
                return;
            }
            rules.forEach(rule -> {
                // 判断本sku是否包含本次修改的选项
                if (rule.getId() == ruleId) {
                    if (existsOtherSellout(skuVo, specs, ruleId)) {
                        // 除本次修改选项外，本商品还有其它售罄选项
                        return;
                    }
                    skuIds.add(skuVo.getSkuId().toString());
                }
            });
        });

        int i = shopGoodsSkuMapper.updateSkuStatus(skuIds, "0", String.valueOf(shopId), String.valueOf(state));
        //推送给极光
        List<Machine> machines = machineMapper.selectMachineByShopId(shopId);
        if (CollectionUtils.isEmpty(machines)) {
            return true;
        }

        List<Long> longSkuIds = skuIds.stream().map(x -> Long.parseLong(x)).collect(Collectors.toList());
        Machine machine = machines.get(0);
        Map<Long, List<Long>> map = new HashMap<>();
        map.put(goodsId, longSkuIds);
        machineApiService.push(machine, String.valueOf(state), map);
        return true;
    }

    /** 判断除本次修改选项外，本商品还有其它售罄选项 */
    private boolean existsOtherSellout(SkuVo skuVo, List<Spec> specs, long ruleId) {
        List<Long> selloutRuleIds = new ArrayList<>();
        for (Spec spec : specs) {
            for (SpecRule rule : spec.getSpecRules()) {
                if (rule.getState() == 1) {
                    continue;
                }
                if (rule.getId() == ruleId) {
                    continue;
                }
                selloutRuleIds.add(rule.getId());
            }
        }

        // 没有其它售罄选项
        if (CollectionUtils.isEmpty(selloutRuleIds)) {
            return false;
        }

        // 有其它售罄选项，判断本商品是否包含其它售罄选项
        List<SpecRule> rules = JSON.parseArray(skuVo.getRulesString(), SpecRule.class);
        for (SpecRule rule : rules) {
            if (selloutRuleIds.contains(rule.getId())) {
                return true;
            }
        }
        return false;
    }
}
