package com.soss.web.controller.coffee;

import com.alibaba.fastjson.JSONObject;
import com.soss.common.annotation.Log;
import com.soss.common.constant.HttpStatus;
import com.soss.common.core.controller.BaseController;
import com.soss.common.core.domain.AjaxResult;
import com.soss.common.core.domain.model.LoginUser;
import com.soss.common.core.page.TableDataInfo;
import com.soss.common.enums.BusinessType;
import com.soss.common.enums.MachineState;
import com.soss.common.exception.ServiceException;
import com.soss.framework.web.service.TokenService;
import com.soss.framework.web.service.WeixinServiceImpl;
import com.soss.system.constants.OrderStatusConstant;
import com.soss.system.domain.Machine;
import com.soss.system.domain.Order;
import com.soss.system.domain.vo.CouponVo;
import com.soss.system.domain.vo.OrderQuery;
import com.soss.system.service.ICouponUserService;
import com.soss.system.service.IGoodsService;
import com.soss.system.service.IMachineService;
import com.soss.system.service.impl.OrderServiceImpl;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletRequest;
import java.math.BigDecimal;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Objects;

/**
 * 订单Controller
 *
 * @author zxq
 * @date 2022-04-28
 */
@RestController
@Slf4j
@RequestMapping("/order")
public class OrderController extends BaseController {
    @Autowired
    private OrderServiceImpl orderService;
    @Autowired
    private IMachineService machineService;
    @Autowired
    private TokenService tokenService;
    @Autowired
    private WeixinServiceImpl weixinService;
    @Autowired
    private ICouponUserService couponUserService;
    @Autowired
    private IGoodsService goodsService;

    /**
     * 查询订单列表
     */
    @PreAuthorize("@ss.hasPermi('system:order:list')")
    @GetMapping("/list")
    public TableDataInfo list(OrderQuery orderQuery) {
        startPage();
        List<Order> list = orderService.selectOrderList(orderQuery);
        orderService.wrapperOrder(list, false);
        return getDataTable(list);
    }


    /**
     * 获取订单详细信息，供小程序使用
     */
    @GetMapping(value = "/{id}")
    public AjaxResult getOrderDetailForApplet(@PathVariable("id") Long id) {
        Order order = orderService.selectOrderById(id);
        goodsService.resolverGoodsOfOrder(order);
        orderService.wrapperOrderDetailForApplet(order);
        return AjaxResult.success(order);
    }

    /**
     * 获取订单详细信息，供后台使用
     */
    @GetMapping(value = "/back/{id}")
    public AjaxResult getOrderDetailForBack(@PathVariable("id") Long id) {
        Order order = orderService.selectOrderById(id);
        goodsService.resolverGoodsOfOrder(order);
        return AjaxResult.success(order);
    }


    @PostMapping("/check")
    public AjaxResult check(@RequestBody Order order) {
        return orderService.check(order);
    }

    /**
     * 新增订单
     */
    @Log(title = "订单", businessType = BusinessType.INSERT)
    @PostMapping
    public AjaxResult add(HttpServletRequest request, @RequestBody Order order) {
        LoginUser loginUser = tokenService.getLoginUser(request);
        if (loginUser == null) {
            throw new ServiceException("请先登录");
        }

        if (order.getLng() == null || order.getLng().compareTo(BigDecimal.ZERO) == 0
                || order.getLat() == null || order.getLat().compareTo(BigDecimal.ZERO) == 0) {
            return AjaxResult.error("未开启微信软件地理位置权限，请前往系统设置中打开");
        }

        if (order.getSendMsgFlag1() == null || order.getSendMsgFlag2() == null || order.getSendMsgFlag3() == null) {
            return AjaxResult.error("授权发送消息权限未传递");
        }

        AjaxResult result = check(order);
        if (HttpStatus.SUCCESS != (int) result.get(result.CODE_TAG)) {
            return result;
        }

        try {
            orderService.checkShopInfo(order);
        } catch (ServiceException e) {
            return AjaxResult.error(e.getMessage());
        }

        if (StringUtils.isEmpty(order.getMachineId())) {
            Machine machine = new Machine();
            machine.setShopId(order.getShopId());
            List<Machine> machines = machineService.selectMachineList(machine);
            if (machines != null && !machines.isEmpty()) {
                Machine machine1 = machines.get(0);
                if (!MachineState.RUNNING.getState().equals(machine1.getState()) && !MachineState.TO_BE_TEST.getState().equals(machine1.getState())) {
                    return AjaxResult.error("机器暂时不可用");
                }
                String machineId = machine1.getId();
                order.setMachineId(machineId);
            } else {
                return AjaxResult.error("该店铺暂无绑定机器");
            }
        }

        Integer couponUserId = order.getCouponId();
        if (couponUserId != null) {
            List<CouponVo> couponVos = couponUserService.listCustAvailableCoupon(loginUser.getOpenId(), couponUserId);
            if (CollectionUtils.isEmpty(couponVos)) {
                return AjaxResult.error("未查询到匹配的优惠券信息");
            }
            couponUserService.resovleCouponFitable(couponVos, order);
            CouponVo couponVo = couponVos.get(0);
            String notFitableDesc = couponVo.getNotFitableDesc();
            if (!StringUtils.isEmpty(notFitableDesc)) {
                return AjaxResult.error(notFitableDesc);
            }
            order.setCouponAmount(couponVo.getCouponAmount().negate());
            order.setPaidAmount(order.getAmount().add(order.getCouponAmount()));
            order.setCouponId(couponVo.getCouponId());
        } else {
            couponUserService.orderDetailCheckAndAssign(order.getShopId(), order.getOrderDetails());
        }

        AjaxResult ajaxResult = orderService.insertOrder(loginUser, order, couponUserId);
        if (HttpStatus.SUCCESS != (int) ajaxResult.get(AjaxResult.CODE_TAG)) {
            return ajaxResult;
        }

        int amount = order.getPaidAmount().movePointRight(2).intValue();
        if (amount == 0) {
            order.setPickCode(weixinService.generatePickCode(order));
            order.setPayTime(new Date());
            order.setState(OrderStatusConstant.Paid);
            orderService.updateOrder(order);
            return AjaxResult.success(order.getId());
        }

        Map payInfo = weixinService.pay(request, "hooloo", order.getOrderNo(), order.getUserId(), amount);
        if (payInfo != null) {
            payInfo.put("orderId", order.getId());
            order.setPayParam(JSONObject.toJSONString(payInfo));
            orderService.updateById(order);
            return AjaxResult.success(payInfo);
        } else {
            throw new ServiceException("微信支付发生异常");
        }
    }

    @RequestMapping("/payOrder")
    public AjaxResult cancel(HttpServletRequest request, Long orderId) {
        Order order = orderService.selectOrderById(orderId);
        Assert.notNull(order, "order not exists");
        Assert.isTrue(Objects.equals(order.getState(), OrderStatusConstant.Unpaid), "订单当前状态不支持该操作");
        if (StringUtils.hasText(order.getPayParam())) {
            return AjaxResult.success(JSONObject.parseObject(order.getPayParam()));
        }

        // order.setOrderNo(orderService.getOrderNo(order.getSource()));
        // orderService.updateOrder(order);
        int amount = order.getPaidAmount().movePointRight(2).intValue();
        if (amount == 0) {
            return AjaxResult.success();
        }
        Map payInfo = weixinService.pay(request, "hooloo", order.getOrderNo(), order.getUserId(), amount);
        if (payInfo != null) {
            return AjaxResult.success(payInfo);
        } else {
            throw new ServiceException("微信支付发生异常");
        }
    }

    /**
     * 修改订单
     */
    @PreAuthorize("@ss.hasPermi('system:order:edit')")
    @Log(title = "订单", businessType = BusinessType.UPDATE)
    @PutMapping
    public AjaxResult edit(@RequestBody Order order) {
        return toAjax(orderService.updateOrder(order));
    }

    @RequestMapping("/cancel")
    public AjaxResult cancel(Long orderId) {
        return toAjax(orderService.cancel(orderId));
    }
}
