package com.soss.web.controller.coupon;


import com.soss.common.annotation.Log;
import com.soss.common.core.controller.BaseController;
import com.soss.common.core.domain.AjaxResult;
import com.soss.common.core.page.TableDataInfo;
import com.soss.common.enums.BusinessType;
import com.soss.system.domain.CouponCategory;
import com.soss.system.service.ICouponCategoryService;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.web.bind.annotation.*;

import java.util.List;

/**
 * <p>
 * 优惠券类别 前端控制器
 * </p>
 *
 * @author caiyt
 * @since 2022-07-21
 */
@RestController
@RequestMapping("/coupon-category")
@Api(tags = "优惠券类别服务类")
public class CouponCategoryController extends BaseController {
    @Autowired
    private ICouponCategoryService couponCategoryService;

    /**
     * 新增优惠券类别
     */
    @PreAuthorize("@ss.hasPermi('coupon:category:add')")
    @Log(title = "新增优惠券类别", businessType = BusinessType.INSERT)
    @PutMapping
    @ApiOperation("新增优惠券类别")
    public AjaxResult add(@RequestBody CouponCategory couponCategory) {
        couponCategoryService.insertCouponCategory(couponCategory);
        return AjaxResult.success();
    }

    /**
     * 更新优惠券类别
     */
    @PreAuthorize("@ss.hasPermi('coupon:category:update')")
    @Log(title = "更新优惠券类别", businessType = BusinessType.UPDATE)
    @PostMapping("update")
    @ApiOperation("更新优惠券类别")
    public AjaxResult update(@RequestBody CouponCategory couponCategory) {
        couponCategoryService.updateCouponCategory(couponCategory);
        return AjaxResult.success();
    }

    /**
     * 查询优惠券类别列表
     */
    @PreAuthorize("@ss.hasPermi('coupon:category:list')")
    @PostMapping
    @ApiOperation("查询优惠券类别列表")
    public TableDataInfo listCouponCategory(@RequestBody CouponCategory couponCategory) {
        startPage();
        List<CouponCategory> couponCategories = couponCategoryService.listCouponCategory(couponCategory);
        return getDataTable(couponCategories);
    }

    /**
     * 上架优惠券类别状态
     */
    @PreAuthorize("@ss.hasPermi('coupon:category:online')")
    @Log(title = "上架优惠券类别", businessType = BusinessType.UPDATE)
    @GetMapping("/online/{id}")
    @ApiOperation("上架优惠券类别状态")
    public AjaxResult switchState(@PathVariable Integer id) {
        int updateCnt = couponCategoryService.onlineCouponCategory(id);
        return toAjax(updateCnt);
    }

    /**
     * 下架优惠券类别状态
     */
    @PreAuthorize("@ss.hasPermi('coupon:category:offline')")
    @Log(title = "下架优惠券类别", businessType = BusinessType.UPDATE)
    @GetMapping("/offline/{id}")
    @ApiOperation("下架优惠券类别状态")
    public AjaxResult offlineCouponCategory(@PathVariable Integer id) {
        int updateCnt = couponCategoryService.offlineCouponCategory(id);
        return toAjax(updateCnt);
    }

    /**
     * 删除优惠券类别状态
     */
    @PreAuthorize("@ss.hasPermi('coupon:category:delete')")
    @Log(title = "删除优惠券类别", businessType = BusinessType.DELETE)
    @DeleteMapping("/{id}")
    @ApiOperation("删除优惠券类别状态")
    public AjaxResult deleteCouponCategory(@PathVariable Integer id) {
        int updateCnt = couponCategoryService.deleteCouponCategory(id);
        return toAjax(updateCnt);
    }
}
