package com.soss.web.controller.coupon;


import com.github.pagehelper.PageInfo;
import com.soss.common.annotation.Log;
import com.soss.common.core.controller.BaseController;
import com.soss.common.core.domain.AjaxResult;
import com.soss.common.core.page.TableDataInfo;
import com.soss.common.enums.BusinessType;
import com.soss.common.enums.CouponCategoryType;
import com.soss.common.enums.CouponState;
import com.soss.common.utils.bean.BeanUtils;
import com.soss.system.domain.Coupon;
import com.soss.system.domain.CouponRule;
import com.soss.system.domain.po.CouponListPo;
import com.soss.system.domain.po.CouponPo;
import com.soss.system.domain.vo.CouponVo;
import com.soss.system.service.ICouponRuleService;
import com.soss.system.service.ICouponService;
import com.soss.system.utils.ArrayUtil;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.util.CollectionUtils;
import org.springframework.web.bind.annotation.*;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * <p>
 * 优惠券 前端控制器
 * </p>
 *
 * @author caiyt
 * @since 2022-07-21
 */
@RestController
@Api(tags = "优惠券服务类")
@RequestMapping("/coupon")
public class CouponController extends BaseController {
    @Autowired
    private ICouponService couponService;
    @Autowired
    private ICouponRuleService couponRuleService;

    /**
     * 查询优惠券列表
     */
    @PreAuthorize("@ss.hasPermi('coupon:list')")
    @PostMapping("list")
    @ApiOperation("查询优惠券列表")
    public TableDataInfo listCoupon(@RequestBody CouponListPo couponListPo) {
        startPage();
        Coupon couponParam = new Coupon();
        BeanUtils.copyProperties(couponListPo, couponParam);
        List<Coupon> coupons = couponService.list(couponParam);
        Map<Integer, CouponRule> ruleMap;
        if (CollectionUtils.isEmpty(coupons)) {
            ruleMap = new HashMap<>();
        } else {
            List<Integer> ruleIds = coupons.stream().map(Coupon::getRuleId).collect(Collectors.toList());
            ruleMap = couponRuleService.getCouponRuleMap(ruleIds);
        }
        List<CouponVo> voList = coupons.stream().map(coupon -> {
            CouponVo couponVo = new CouponVo();
            BeanUtils.copyProperties(coupon, couponVo);
            couponVo.setStateDesc(CouponState.getDesc(coupon.getState()));
            couponVo.setTypeDesc(CouponCategoryType.getDesc(coupon.getType()));
            CouponRule couponRule = ruleMap.get(coupon.getRuleId());
            if (couponRule != null) {
                transLimitToArrStyle(couponVo, couponRule);
                couponVo.setRuleId(coupon.getRuleId());
                couponVo.setRuleName(couponRule.getName());
                couponVo.setRuleDesc(couponRule.getDesc());
                if (Objects.equals(couponRule.getDaysLimit(), 0)) {
                    couponVo.setDaysLimit(null);
                }
            }
            return couponVo;
        }).collect(Collectors.toList());
        couponRuleService.translateUseLimit(voList);
        TableDataInfo dataTable = getDataTable(voList);
        dataTable.setTotal(new PageInfo(coupons).getTotal());
        return dataTable;
    }

    private void transLimitToArrStyle(CouponVo couponVo, CouponRule couponRule) {
        BeanUtils.copyProperties(couponRule, couponVo, "name", "state", "categoryIds", "goodsIds", "province", "city", "area", "shopIds");
        couponVo.setCategoryIds(ArrayUtil.transStrToLongList(couponRule.getCategoryIds()));
        couponVo.setGoodsIds(ArrayUtil.transStrToLongList(couponRule.getGoodsIds()));
        couponVo.setProvince(ArrayUtil.transStrToCodeList(couponRule.getProvince()));
        couponVo.setCity(ArrayUtil.transStrToCodeList(couponRule.getCity()));
        couponVo.setArea(ArrayUtil.transStrToCodeList(couponRule.getArea()));
        couponVo.setShopIds(ArrayUtil.transStrToLongList(couponRule.getShopIds()));
        couponVo.setWeekLimit(ArrayUtil.transStrToIntList(couponRule.getWeekLimit()));
    }

    /**
     * 获取优惠券详情
     */
    @PreAuthorize("@ss.hasPermi('coupon:detail')")
    @GetMapping("/detail/{id}")
    @ApiOperation("获取优惠券详情")
    public AjaxResult detail(@PathVariable Integer id) {
        Coupon coupon = couponService.detail(id);
        CouponVo couponVo = new CouponVo();
        BeanUtils.copyProperties(coupon, couponVo);
        CouponRule couponRule = couponRuleService.detail(coupon.getRuleId());
        transLimitToArrStyle(couponVo, couponRule);
        if (Objects.equals(couponRule.getDaysLimit(), 0)) {
            couponVo.setDaysLimit(null);
        }
        return AjaxResult.success(couponVo);
    }

    /**
     * 新增优惠券
     */
    @PreAuthorize("@ss.hasPermi('coupon:add')")
    @Log(title = "新增优惠券", businessType = BusinessType.INSERT)
    @PutMapping
    @ApiOperation("新增优惠券")
    public AjaxResult save(@RequestBody CouponPo couponPo) {
        couponService.save(couponPo);
        return AjaxResult.success();
    }

    /**
     * 更新优惠券
     */
    @PreAuthorize("@ss.hasPermi('coupon:update')")
    @Log(title = "更新优惠券", businessType = BusinessType.UPDATE)
    @PostMapping
    @ApiOperation("更新优惠券")
    public AjaxResult update(@RequestBody CouponPo couponPo) {
        couponService.update(couponPo);
        return AjaxResult.success();
    }

    /**
     * 更新优惠券预校验
     */
    @PreAuthorize("@ss.hasPermi('coupon:update:check')")
    @PostMapping("/update/check/{id}")
    @ApiOperation("更新优惠券预校验")
    public AjaxResult checkPreUpdate(@PathVariable Integer id) {
        couponService.updateCheckPre(id);
        return AjaxResult.success();
    }

    /**
     * 上架优惠券
     */
    @PreAuthorize("@ss.hasPermi('coupon:online')")
    @Log(title = "上架优惠券", businessType = BusinessType.UPDATE)
    @GetMapping("/online/{id}")
    @ApiOperation("上架优惠券")
    public AjaxResult onlineCoupon(@PathVariable Integer id) {
        int updateCnt = couponService.onlineCoupon(id);
        return toAjax(updateCnt);
    }

    /**
     * 下架优惠券
     */
    @PreAuthorize("@ss.hasPermi('coupon:offline')")
    @Log(title = "下架优惠券", businessType = BusinessType.UPDATE)
    @GetMapping("/offline/{id}")
    @ApiOperation("下架优惠券")
    public AjaxResult offlineCoupon(@PathVariable Integer id) {
        int updateCnt = couponService.offlineCoupon(id);
        return toAjax(updateCnt);
    }

    /**
     * 删除优惠券
     */
    @PreAuthorize("@ss.hasPermi('coupon:delete')")
    @Log(title = "删除优惠券", businessType = BusinessType.DELETE)
    @DeleteMapping("/{id}")
    @ApiOperation("删除优惠券")
    public AjaxResult deleteCoupon(@PathVariable Integer id) {
        int updateCnt = couponService.deleteCoupon(id);
        return toAjax(updateCnt);
    }

    /**
     * 获取当前有效的规则集合
     */
    // @PreAuthorize("@ss.hasPermi('coupon:list:effective')")
    @GetMapping("/list/effective")
    @ApiOperation("获取当前有效的规则集合")
    public TableDataInfo listEffective() {
        startPage();
        List<Coupon> effectiveCoupons = couponService.getEffectiveCoupon();
        return getDataTable(effectiveCoupons);
    }
}
