package com.soss.system.service.impl;

import com.soss.common.core.domain.model.LoginUser;
import com.soss.common.enums.ShopState;
import com.soss.common.utils.DateUtils;
import com.soss.common.utils.StringUtils;
import com.soss.system.domain.Order;
import com.soss.system.domain.Shop;
import com.soss.system.domain.po.LocationPo;
import com.soss.system.domain.vo.LocationVo;
import com.soss.system.mapper.ShopMapper;
import com.soss.system.utils.AreaUtil;
import com.soss.system.utils.DistanceUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.BooleanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.text.ParseException;
import java.util.Arrays;
import java.util.Comparator;
import java.util.List;

@Service
@Slf4j
public class AppServiceImpl {
    @Autowired
    private ShopMapper shopMapper;
    @Autowired
    private ShopServiceImpl shopService;
    @Autowired
    private OrderServiceImpl orderService;
    @Autowired
    private AreaUtil areaUtil;

    public Shop getShop(LoginUser loginUser, String lng, String lat, Boolean testFlag) throws ParseException {
        Shop shop = null;
        if (StringUtils.isEmpty(lng) || StringUtils.isEmpty(lat)) {
            log.info("getShop lng&lat is null");
            // 未授权位置
            if (loginUser == null) {
                // 未登录，返回默认店铺
                shop = getDefaultShop();
            } else {
                // 已登录
                List<Order> myOrder = orderService.getMyOrder(loginUser.getOpenId());
                if (CollectionUtils.isEmpty(myOrder)) {
                    // 未下过单返回默认店铺
                    shop = getDefaultShop();
                } else {
                    // 返回最后下单店铺
                    Order order = myOrder.get(0);
                    shop = shopService.selectShopById(order.getShopId());
                }
            }
        } else {
            // 授权了位置
            log.info("getShop lng={}, lat={}", lng, lat);
            List<Integer> states = null;
            if (BooleanUtils.isTrue(testFlag)) {
                states = Arrays.asList(ShopState.TESTING.getState(), ShopState.OPEN.getState());
            } else {
                states = Arrays.asList(ShopState.OPEN.getState(), ShopState.CLOSE.getState());
            }

            List<Shop> shops = shopMapper.selectEnableShops(states); // 获取所有非关闭的店铺
            if (CollectionUtils.isEmpty(shops)) {
                return null;
            }

            for (Shop s : shops) { // 获取所有店铺距离
                double realDistance = DistanceUtil.getRealDistance(Double.parseDouble(lng), Double.parseDouble(lat), Double.parseDouble(s.getLng()), Double.parseDouble(s.getLat()));
                s.setRealDistance(realDistance);
            }
            shops.sort(Comparator.comparing(Shop::getRealDistance)); // 按距离排序
            for (Shop s : shops) {
                if (perfectOrderState(s, lng, lat)) { // 是否能下单
                    shop = s;
                    break;
                }
            }

            if (shop == null) { // 没有能下单店铺选择最近的店铺
                shop = shops.get(0);
            }
        }
        if (shop.getOrderState() == null) {
            perfectOrderState(shop, lng, lat);
        }
        return shop;
    }

    public boolean perfectOrderState(Shop shop, String lng, String lat) throws ParseException {
        if (shop == null) {
            return false;
        }

        // 状态已停止营业
        if (shop.getState() != 1) {
            shop.setOrderState(0);
            shop.setStateDesc("门店已暂停营业");
            return false;
        }

        // 不在营业时间范围内
        if (!DateUtils.isEffectiveDate(shop.getStartTime(), shop.getEndTime())) {
            shop.setOrderState(0);
            shop.setStateDesc("门店已休息");
            return false;
        }

        // 超出最大配送距离10KM 或 超出店铺设置最大距离
        if (!StringUtils.isEmpty(lng) && !StringUtils.isEmpty(lat)) {
            double realDistance = DistanceUtil.getRealDistance(Double.parseDouble(lng), Double.parseDouble(lat), Double.parseDouble(shop.getLng()), Double.parseDouble(shop.getLat()));
            double realKm = realDistance / 1000;
//            if (realKm > 10) {
//                shop.setOrderState(0);
//                shop.setStateDesc("距离较远，门店暂停接单");
//                return false;
//            }
            Double shopLimit = shop.getDistanceLimit();
            if (shopLimit != null && shopLimit > 0 && realKm > shopLimit) {
                shop.setOrderState(0);
                shop.setStateDesc("距离较远，门店暂停接单");
                return false;
            }
        }

        shop.setOrderState(1);
        return true;
    }

    public List<Shop> getSameLocationShops(LoginUser loginUser, LocationPo locationPo, Boolean testFlag) throws ParseException {
        List<Integer> states = null;
        if (BooleanUtils.isTrue(testFlag)) {
            states = Arrays.asList(ShopState.TESTING.getState(), ShopState.OPEN.getState());
        } else {
            states = Arrays.asList(ShopState.OPEN.getState(), ShopState.CLOSE.getState());
        }

        List<Shop> shops = null;
        if (locationPo == null || locationPo.isEmpty()) {
            // 未授权位置，返回默认店铺相同区域的所有店铺
            shops = shopMapper.selectDefaultSameLocationShops(states);
        } else {
            // 授权了位置
            LocationVo locationVo = areaUtil.translateLocation(locationPo);
            if (locationVo == null) {
                shops = shopMapper.selectDefaultSameLocationShops(states);
            } else {
                shops = shopMapper.selectSameLocationShops(states, locationVo);
            }
        }

        if (CollectionUtils.isEmpty(shops)) {
            return null;
        }

        for (Shop s : shops) {
            if (locationPo == null || locationPo.isEmpty()) {
                s.setRealDistance(0d);
                s.setDistance("-1");
                s.setOrderState(1);
            } else {
                // 获取所有店铺距离
                double realDistance = DistanceUtil.getRealDistance(Double.parseDouble(locationPo.getLng()), Double.parseDouble(locationPo.getLat()), Double.parseDouble(s.getLng()), Double.parseDouble(s.getLat()));
                s.setRealDistance(realDistance);
                s.setDistance(DistanceUtil.getDistanceDesc(realDistance));
                perfectOrderState(s, locationPo.getLng(), locationPo.getLat());
            }
        }
        shops.sort(Comparator.comparing(Shop::getRealDistance)); // 按距离排序
        return shops;
    }

    public Shop getShop(LoginUser loginUser) {
        if (loginUser == null) {
            // 未登录，返回默认店铺
            return getDefaultShop();
        } else {
            // 已登录
            List<Order> myOrder = orderService.getMyOrder(loginUser.getOpenId());
            if (CollectionUtils.isEmpty(myOrder)) {
                // 未下过单返回默认店铺
                return getDefaultShop();
            }
            // 返回最后下单店铺
            Order order = myOrder.get(0);
            Shop shop = shopService.selectShopById(order.getShopId());
            return shop == null ? new Shop() : shop;
        }
    }

    public Shop getDefaultShop() {
        Shop shop = new Shop();
        shop.setIsDefault(1L);
        List<Shop> shops = shopMapper.selectShopList(shop);
        return CollectionUtils.isEmpty(shops) ? new Shop() : shops.get(0);
    }

    /**
     * 授权了地理位置的用户都展示距离最近的营业中的店铺；
     * 未授权地理位置的用户都展示默认店铺，不管店铺是否营业中；
     */
    public Shop getShop(String lng, String lat, Boolean testFlag) {
        Shop shop = new Shop();

        Comparator<Shop> shopComparator = Comparator.comparing(Shop::getRealDistance);
        if (BooleanUtils.isTrue(testFlag)) {
            shop.setStates(Arrays.asList(ShopState.TESTING.getState(), ShopState.OPEN.getState()));
        } else {
            shop.setStates(Arrays.asList(ShopState.OPEN.getState(), ShopState.CLOSE.getState()));
            shopComparator = Comparator.comparing(Shop::getState).thenComparing(Shop::getRealDistance);
        }

        List<Shop> shops = shopMapper.selectShopList(shop);
        if (CollectionUtils.isEmpty(shops)) {
            return new Shop();
        }

        shops.forEach(sp -> {
            double realDistance = DistanceUtil.getRealDistance(Double.parseDouble(lng), Double.parseDouble(lat), Double.parseDouble(sp.getLng()), Double.parseDouble(sp.getLat()));
            sp.setRealDistance(realDistance);
            String distanceDesc = DistanceUtil.getDistanceDesc(realDistance);
            sp.setDistance(distanceDesc);
        });
        shops.sort(shopComparator);
        return shops.get(0);
    }
}
