package com.soss.system.service.impl;

import com.soss.common.exception.ServiceException;
import com.soss.common.utils.StringUtils;
import com.soss.system.domain.Shop;
import com.soss.system.domain.SysBanner;
import com.soss.system.domain.po.BannerListPo;
import com.soss.system.domain.po.BannerPo;
import com.soss.system.domain.vo.BannerVo;
import com.soss.system.mapper.ShopMapper;
import com.soss.system.mapper.SysBannerMapper;
import com.soss.system.service.ISysBannerService;
import com.soss.system.utils.AreaUtil;
import com.soss.system.utils.ArrayUtil;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 轮播图Service业务层处理
 *
 * @author zxq
 * @date 2022-04-28
 */
@Service
@Slf4j
public class SysBannerServiceImpl implements ISysBannerService {
    @Autowired
    private SysBannerMapper sysBannerMapper;
    @Autowired
    private ShopServiceImpl shopService;
    @Autowired
    private ShopMapper shopMapper;
    @Autowired
    private AreaUtil areaUtil;

    @Override
    public List<SysBanner> selectBanner(BannerListPo banner) {
        return sysBannerMapper.selectBanner(banner);
    }

    @Override
    public Long createBanner(BannerPo bannerPo) {
        SysBanner banner = copyBanner(bannerPo);
        if (StringUtils.isEmpty(banner.getImage())) {
            throw new ServiceException("图片地址不能为空");
        }
//        if (StringUtils.isEmpty(banner.getPage())) {
//            throw new ServiceException("链接地址不能为空");
//        }
        if (banner.getType() == null || banner.getType() < 1 || banner.getType() > 3) {
            throw new ServiceException("类型不正确，1-小程序，2-KDS，3-点单屏");
        }
        if (StringUtils.isEmpty(banner.getTarget())) {
            throw new ServiceException("轮播位置不正确，1-小程序首页，2-小程序我的，0-其它");
        }
        if (banner.getSort() == null) {
            throw new ServiceException("轮播顺序不能为空");
        }
        if (StringUtils.isEmpty(banner.getProvince()) && StringUtils.isEmpty(banner.getCity()) && StringUtils.isEmpty(banner.getArea()) && StringUtils.isEmpty(banner.getShopIds())) {
            throw new ServiceException("适用范围不能为空");
        }

        banner.setCreatedAt(new Date());
        banner.setUpdatedAt(new Date());
        banner.setState(0);
        long i = sysBannerMapper.createBanner(banner);
        return banner.getId();
    }

    @Override
    public Boolean updateBanner(BannerPo bannerPo) {
        SysBanner banner = copyBanner(bannerPo);
        if (banner.getId() == null || banner.getId() <= 0) {
            throw new ServiceException("id不能为空");
        }
        if (StringUtils.isEmpty(banner.getImage())) {
            throw new ServiceException("图片地址不能为空");
        }
//        if (StringUtils.isEmpty(banner.getPage())) {
//            throw new ServiceException("链接地址不能为空");
//        }
        if (banner.getType() == null || banner.getType() < 1 || banner.getType() > 3) {
            throw new ServiceException("类型不正确，1-小程序，2-KDS，3-点单屏");
        }
        if (StringUtils.isEmpty(banner.getTarget())) {
            throw new ServiceException("轮播位置不正确，1-小程序首页，2-小程序我的，0-其它");
        }
        if (banner.getSort() == null) {
            throw new ServiceException("轮播顺序不能为空");
        }
        if (StringUtils.isEmpty(banner.getProvince()) && StringUtils.isEmpty(banner.getCity()) && StringUtils.isEmpty(banner.getArea()) && StringUtils.isEmpty(banner.getShopIds())) {
            throw new ServiceException("适用范围不能为空");
        }

        banner.setUpdatedAt(new Date());

        long i = sysBannerMapper.updateBanner(banner);
        return i > 0 ? true : false;
    }

    private SysBanner copyBanner(BannerPo bannerPo) {
        SysBanner banner = new SysBanner();
        banner.setId(bannerPo.getId());
        banner.setImage(bannerPo.getImage());
        banner.setPage(bannerPo.getPage());
        banner.setType(bannerPo.getType());
        banner.setSort(bannerPo.getSort());
        banner.setState(bannerPo.getState());
//        banner.setTarget(bannerPo.getTarget());
        if (bannerPo.getTarget() != null && bannerPo.getTarget().length > 0) {
            banner.setTarget("," + Arrays.asList(bannerPo.getTarget()).stream().collect(Collectors.joining(",")) + ",");
        }
        if (bannerPo.getProvince() != null && bannerPo.getProvince().length > 0) {
            banner.setProvince("," + Arrays.asList(bannerPo.getProvince()).stream().collect(Collectors.joining(",")) + ",");
        }
        if (bannerPo.getCity() != null && bannerPo.getCity().length > 0) {
            banner.setCity("," + Arrays.asList(bannerPo.getCity()).stream().collect(Collectors.joining(",")) + ",");
        }
        if (bannerPo.getArea() != null && bannerPo.getArea().length > 0) {
            banner.setArea("," + Arrays.asList(bannerPo.getArea()).stream().collect(Collectors.joining(",")) + ",");
        }
        if (bannerPo.getShopIds() != null && bannerPo.getShopIds().length > 0) {
            banner.setShopIds("," + Arrays.asList(bannerPo.getShopIds()).stream().map(x -> String.valueOf(x)).collect(Collectors.joining(",")) + ",");
        }
        return banner;
    }

    private BannerVo copyBanner(SysBanner banner) {
        BannerVo bannerVo = new BannerVo();
        bannerVo.setId(banner.getId());
        bannerVo.setImage(banner.getImage());
        bannerVo.setPage(banner.getPage());
        bannerVo.setType(banner.getType());
        bannerVo.setSort(banner.getSort());
        bannerVo.setState(banner.getState());

//        bannerVo.setTarget(banner.getTarget());
        if (!StringUtils.isEmpty(banner.getTarget())) {
            String str = trim(banner.getTarget());
            bannerVo.setTarget(str.split(","));
            String targetStr = "";
            for (String s : bannerVo.getTarget()) {
                if (!StringUtils.isEmpty(targetStr)) {
                    targetStr += ",";
                }
                if (s.equals("1")) {
                    targetStr += "菜单页顶部";
                } else if (s.equals("2")) {
                    targetStr += "我的页banner";
                } else if (s.equals("0")) {
                    targetStr += "菜单轮播";
                }
            }
            bannerVo.setTargetStr(targetStr);
        }
        if (!StringUtils.isEmpty(banner.getProvince())) {
            String str = trim(banner.getProvince());
            bannerVo.setProvinceStr(str);
            bannerVo.setProvince(str.split(","));
        }
        if (!StringUtils.isEmpty(banner.getCity())) {
            String str = trim(banner.getCity());
            bannerVo.setCityStr(str);
            bannerVo.setCity(str.split(","));
        }
        if (!StringUtils.isEmpty(banner.getArea())) {
            String str = trim(banner.getArea());
            bannerVo.setAreaStr(str);
            bannerVo.setArea(str.split(","));
        }
        if (!StringUtils.isEmpty(banner.getShopIds())) {
            String str = trim(banner.getShopIds());
            bannerVo.setShopIdsStr(str);
            String[] shopIds = str.split(",");
            Long[] ids = new Long[shopIds.length];
            for (int i = 0; i < shopIds.length; i++) {
                ids[i] = Long.parseLong(shopIds[i]);
            }
            bannerVo.setShopIds(ids);
        }
        return bannerVo;
    }

    private String trim(String str) {
        if (StringUtils.isEmpty(str)) {
            return str;
        }
        return str.substring(1, str.length() - 1);
    }

    public List<BannerVo> copyBanner(List<SysBanner> banners) {
        List<BannerVo> bannerVos = new ArrayList<>();
        if (CollectionUtils.isEmpty(banners)) {
            return bannerVos;
        }
        banners.forEach(banner -> bannerVos.add(copyBanner(banner)));
        return bannerVos;
    }

    @Override
    public Boolean updateState(Long bannerId, Integer state) {
        SysBanner banner = new SysBanner();
        banner.setId(bannerId);
        banner.setState(state);
        banner.setUpdatedAt(new Date());

        if (banner.getId() == null || banner.getId() <= 0) {
            throw new ServiceException("id不能为空");
        }
        if (banner.getState() == null || banner.getState() < 0 || banner.getState() > 1) {
            throw new ServiceException("状态不正确，0-上线，1-下线");
        }

        long i = sysBannerMapper.updateState(banner);
        return i > 0 ? true : false;
    }

    @Override
    public Boolean delete(Long bannerId) {
        long i = sysBannerMapper.delete(bannerId);
        return i > 0 ? true : false;
    }

    @Override
    public List<SysBanner> getBannersByShopId(Long shopId, Integer type, Integer target) {
        List<SysBanner> banners = sysBannerMapper.selectBanners(type, target);
        if (CollectionUtils.isEmpty(banners)) {
            log.info("banner list is empty");
            return null;
        }
        Shop shop = shopService.selectShopById(shopId);
        if (shop == null) {
            log.info("shop is null");
            return null;
        }
        List<SysBanner> list = new ArrayList<>();
        banners.forEach(banner -> {
            if (!StringUtils.isEmpty(banner.getProvince())) {
                String province = trim(banner.getProvince());
                if (province.equals("0")) {
                    list.add(banner);
                    return;
                }
                Arrays.stream(province.split(",")).forEach(p -> {
                    if (p.equals(shop.getProvince())) {
                        list.add(banner);
                        return;
                    }
                });
            }
            if (!StringUtils.isEmpty(banner.getCity())) {
                String city = trim(banner.getCity());
                Arrays.stream(city.split(",")).forEach(p -> {
                    if (p.equals(shop.getCity())) {
                        list.add(banner);
                        return;
                    }
                });
            }
            if (!StringUtils.isEmpty(banner.getArea())) {
                String area = trim(banner.getArea());
                Arrays.stream(area.split(",")).forEach(p -> {
                    if (p.equals(shop.getZone())) {
                        list.add(banner);
                        return;
                    }
                });
            }
            if (!StringUtils.isEmpty(banner.getShopIds())) {
                String shopIds = trim(banner.getShopIds());
                Arrays.stream(shopIds.split(",")).forEach(p -> {
                    if (p.equals(String.valueOf(shop.getId()))) {
                        list.add(banner);
                        return;
                    }
                });
            }
        });
        return list;
    }

    @Override
    public List<String> getKDSBannerByShopId(Long shopId) {
        List<SysBanner> banners = sysBannerMapper.selectBanners(2, 0);
        if (CollectionUtils.isEmpty(banners)) {
            log.info("banner list is empty");
            return null;
        }
        Shop shop = shopService.selectShopById(shopId);
        if (shop == null) {
            log.info("shop is null");
            return null;
        }
        List<String> list = new ArrayList<>();
        banners.forEach(banner -> {
            if (!StringUtils.isEmpty(banner.getProvince())) {
                String province = trim(banner.getProvince());
                if (province.equals("0")) {
                    list.add(banner.getImage());
                    return;
                }
                Arrays.stream(banner.getProvince().split(",")).forEach(p -> {
                    if (p.equals(shop.getProvince())) {
                        list.add(banner.getImage());
                        return;
                    }
                });
            }
            if (!StringUtils.isEmpty(banner.getCity())) {
                String city = trim(banner.getCity());
                Arrays.stream(city.split(",")).forEach(p -> {
                    if (p.equals(shop.getCity())) {
                        list.add(banner.getImage());
                        return;
                    }
                });
            }
            if (!StringUtils.isEmpty(banner.getArea())) {
                String area = trim(banner.getArea());
                Arrays.stream(area.split(",")).forEach(p -> {
                    if (p.equals(shop.getZone())) {
                        list.add(banner.getImage());
                        return;
                    }
                });
            }
            if (!StringUtils.isEmpty(banner.getShopIds())) {
                String shopIds = trim(banner.getShopIds());
                Arrays.stream(shopIds.split(",")).forEach(p -> {
                    if (p.equals(String.valueOf(shop.getId()))) {
                        list.add(banner.getImage());
                        return;
                    }
                });
            }
        });
        return list;
    }

    public void translateArea(List<BannerVo> bannerVos) {
        if (CollectionUtils.isEmpty(bannerVos)) {
            return;
        }
        List<String> province = new ArrayList<>(16);
        List<String> city = new ArrayList<>(16);
        List<String> area = new ArrayList<>(16);
        List<Long> shopIds = new ArrayList<>(16);
        for (BannerVo bannerVo : bannerVos) {
            if (!ArrayUtil.isAllAvailable(bannerVo.getProvince())) {
                ArrayUtil.listAddArray(province, bannerVo.getProvince());
            }
            ArrayUtil.listAddArray(city, bannerVo.getCity());
            ArrayUtil.listAddArray(area, bannerVo.getArea());
            ArrayUtil.listAddArray(shopIds, bannerVo.getShopIds());
        }

        Map<Long, Shop> shopMapInit;
        if (!CollectionUtils.isEmpty(shopIds)) {
            List<Shop> shops = shopMapper.selectShopByIds(shopIds);
            shopMapInit = shops.stream().collect(Collectors.toMap(Shop::getId, Function.identity()));
        } else {
            shopMapInit = new HashMap<>(0);
        }
        Map<Long, Shop> shopMap = shopMapInit;

        for (BannerVo bannerVo : bannerVos) {
            if (ArrayUtil.isAllAvailable(bannerVo.getProvince())) {
                bannerVo.setAreaLimitDesc("全国通用");
                continue;
            } else if (ArrayUtil.hasContents(bannerVo.getProvince())) {
                String provDesc = Arrays.stream(bannerVo.getProvince()).map(areaUtil::getProvNameByCode).collect(Collectors.joining(","));
                bannerVo.setAreaLimitDesc(provDesc);
                continue;
            }

            if (ArrayUtil.hasContents(bannerVo.getCity())) {
                String cityDesc = Arrays.stream(bannerVo.getCity()).map(areaUtil::getCityOrAreaNameByCode).collect(Collectors.joining(","));
                bannerVo.setAreaLimitDesc(cityDesc);
                continue;
            }

            if (ArrayUtil.hasContents(bannerVo.getArea())) {
                String areaDesc = Arrays.stream(bannerVo.getArea()).map(areaUtil::getCityOrAreaNameByCode).collect(Collectors.joining(","));
                bannerVo.setAreaLimitDesc(areaDesc);
                continue;
            }

            if (!ArrayUtil.hasContents(bannerVo.getShopIds())) {
                continue;
            }

            String shopDesc = Arrays.stream(bannerVo.getShopIds()).map(shopId -> {
                Shop shop = shopMap.get(shopId);
                if (shop == null) {
                    return null;
                }
                return areaUtil.getCityOrAreaNameByCode(shop.getZone()) + shop.getName();
            }).filter(Objects::nonNull).collect(Collectors.joining(","));
            bannerVo.setAreaLimitDesc(shopDesc);
        }
    }
}
