package com.cnooc.expert.service.auth;


import com.cnooc.expert.system.entity.pojo.SlideImageResult;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.io.Resource;
import org.springframework.stereotype.Service;

import javax.imageio.ImageIO;
import java.awt.*;
import java.awt.geom.RoundRectangle2D;
import java.awt.image.BufferedImage;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.Base64;
import java.util.concurrent.ThreadLocalRandom;

@Service
public class SlideCaptchaService {
    private static final int SLIDER_WIDTH = 50;
    private static final int SLIDER_HEIGHT = 50;

    @Value("classpath:images/backgrounds/*")
    private Resource[] backgroundResources;

    /**
     * 使用预设背景图片生成验证码
     */
    public SlideImageResult generateSlideImages() throws IOException {
        // 随机选择背景图片
        Resource bgResource = backgroundResources[
                ThreadLocalRandom.current().nextInt(backgroundResources.length)
                ];

        BufferedImage background = ImageIO.read(bgResource.getInputStream());

        // 调整背景图尺寸
        background = resizeImage(background, 300, 200);

        // 生成滑块位置
        int sliderX = ThreadLocalRandom.current().nextInt(50, 250);
        int sliderY = ThreadLocalRandom.current().nextInt(10, 150);

        // 创建滑块
        BufferedImage sliderImage = createSliderWithShadow(background, sliderX, sliderY);

        drawSliderHole(background, sliderX, sliderY);

        // 转换为Base64
        String bgBase64 = imageToBase64(background, "png");
        String sliderBase64 = imageToBase64(sliderImage, "png");

        SlideImageResult result = new SlideImageResult();
        result.setBackgroundBase64(bgBase64);
        result.setSlideBase64(sliderBase64);
        result.setSliderX(sliderX);
        result.setSliderY(sliderY);

        return result;
    }

    /**
     * 图片转换为Base64
     */
    private String imageToBase64(BufferedImage image, String format) throws IOException {
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        ImageIO.write(image, format, baos);
        byte[] imageBytes = baos.toByteArray();
        return Base64.getEncoder().encodeToString(imageBytes);
    }

    /**
     * 在背景图上绘制滑块挖空效果
     */
    private void drawSliderHoleOld(BufferedImage background, int x, int y) {
        Graphics2D g2d = background.createGraphics();

        // 设置挖空区域
        g2d.setComposite(AlphaComposite.Clear);
        g2d.fillRect(x, y, SLIDER_WIDTH, SLIDER_HEIGHT);

        // 重置合成规则，绘制边框
        g2d.setComposite(AlphaComposite.SrcOver);
        g2d.setColor(Color.GRAY);
        g2d.setStroke(new BasicStroke(2));
        g2d.drawRect(x, y, SLIDER_WIDTH, SLIDER_HEIGHT);

        g2d.dispose();
    }
    /**
     * 在背景图上绘制滑块挖空效果
     */
    private void drawSliderHole(BufferedImage background, int x, int y) {
        Graphics2D g2d = background.createGraphics();
        // 设置挖空区域
        // 创建半透明效果，能看到下层背景
        g2d.setComposite(AlphaComposite.getInstance(AlphaComposite.SRC_OVER, 0.8f)); // 50%透明度
        // 使用灰色填充
        g2d.setColor(new Color(100, 100, 100)); // 深灰色
        g2d.fillRect(x, y, SLIDER_WIDTH, SLIDER_HEIGHT);
        g2d.dispose();
    }
    /**
     * 创建带阴影效果的滑块
     */
    private BufferedImage createSliderWithShadowOld(BufferedImage background, int x, int y) {
        int width = 50;
        int height = 50;
        int shadowSize = 3;

        BufferedImage slider = new BufferedImage(
                width + shadowSize, height + shadowSize, BufferedImage.TYPE_INT_ARGB);
        Graphics2D g2d = slider.createGraphics();

        g2d.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);

        // 绘制阴影
        g2d.setColor(new Color(0, 0, 0, 100));
        g2d.fillRoundRect(shadowSize, shadowSize, width, height, 10, 10);

        // 绘制滑块主体
        Shape sliderShape = new RoundRectangle2D.Double(0, 0, width, height, 10, 10);
        g2d.setClip(sliderShape);

        BufferedImage subImage = background.getSubimage(x, y, width, height);
        g2d.drawImage(subImage, 0, 0, null);

        g2d.setClip(null);

        // 绘制边框
        g2d.setColor(new Color(64, 158, 255));
        g2d.setStroke(new BasicStroke(2));
        g2d.draw(sliderShape);

        g2d.dispose();
        return slider;
    }

    /**
     * 创建带阴影效果的滑块
     */
    private BufferedImage createSliderWithShadow(BufferedImage background, int x, int y) {
        int width = 50;
        int height = 50;

        BufferedImage slider = new BufferedImage(width, height, BufferedImage.TYPE_INT_ARGB);
        Graphics2D g2d = slider.createGraphics();

        // 直接截取背景图部分，不需要复杂的效果
        BufferedImage subImage = background.getSubimage(x, y, width, height);
        g2d.drawImage(subImage, 0, 0, null);

        g2d.dispose();
        return slider;
    }
    /**
     * 调整图片尺寸
     */
    private BufferedImage resizeImage(BufferedImage originalImage, int targetWidth, int targetHeight) {
        BufferedImage resizedImage = new BufferedImage(targetWidth, targetHeight, BufferedImage.TYPE_INT_RGB);
        Graphics2D g2d = resizedImage.createGraphics();

        g2d.setRenderingHint(RenderingHints.KEY_INTERPOLATION, RenderingHints.VALUE_INTERPOLATION_BILINEAR);
        g2d.drawImage(originalImage, 0, 0, targetWidth, targetHeight, null);
        g2d.dispose();

        return resizedImage;
    }
}
