package com.cnooc.expert.auth.service.impl;

import cn.hutool.core.util.RandomUtil;
import com.cnooc.expert.common.constant.TokenConstants;
import com.cnooc.expert.common.utils.SmsHttpUtil;
import com.cnooc.expert.common.utils.SmsUtil;
import com.cnooc.expert.system.entity.vo.SmsConfig;
import com.cnooc.expert.auth.service.SmsService;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.stereotype.Service;
import java.util.concurrent.TimeUnit;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.CompletableFuture;

@Service
@AllArgsConstructor
@Slf4j
public class SmsServiceImpl implements SmsService {
    public static final String SMS_CODE_CONTENT = "福建通信行业融合创新服务平台，您的验证码是：%s（有效期为2分钟），请勿泄露给他人，如非本人操作，请忽略此消息。";
    @Autowired
    private StringRedisTemplate redisTemplate;
    private final SmsConfig smsConfig;

    /**
     * 校验短信验证码
     *
     * @param phone 手机号
     * @param code  用户输入的验证码
     * @return 是否校验通过
     */
    @Override
    public boolean verifySmsCode(String phone, String code) {
        String key = "sms:code:" + phone;
        String storedCode = redisTemplate.opsForValue().get(key);
        if (storedCode != null && storedCode.equals(code)) {
            // 验证码正确，删除缓存中的验证码
            redisTemplate.delete(key);
            return true;
        }
        return false;
    }

    /**
     * 发送短信验证码（可选方法）
     *
     * @param phone 手机号
     * @return 是否发送成功
     */
    public boolean sendSmsCode(String phone) {
        // 生成6位验证码
        //String code = RandomUtil.randomNumbers(6);
        String code = "666666";
        String key = "sms:code:" + phone;
        redisTemplate.opsForValue().set(key, code, 5,TimeUnit.MINUTES);// 存入 Redis，设置过期时间为5分钟
        //String storedCode = (String)redisTemplate.opsForValue().get(key);
        System.out.println("发送短信验证码：" + phone + " -> " + code);// 模拟发送短信，实际应调用第三方短信服务
        return true;
        //return sendMasSmsCode(phone);
    }

    @Override
    public boolean sendMasSmsCode(String phone) {
        // 生成验证码并构建缓存键
        String code = SmsUtil.generateVerificationCode();
        System.out.println("发送短信验证码：" + phone + " -> " + code);
        String key = SmsUtil.buildCacheKey(phone);

        try {
            // 构建短信内容
            String content = String.format(SMS_CODE_CONTENT,  code);
            System.out.println("发送短信验证码：" + phone + " -> " + code);
            log.info("云MAS业务平台 发送手机号: {}，短信验证码：{}", phone, code);
            System.out.println("云MAS业务平台 短信内容: " + content);

            boolean result = sendMasSmsContent(phone, content);
            if(result){
                redisTemplate.opsForValue().set(key, code, 2 * 60); // 存入 Redis，设置过期时间为2分钟
            }
            return result;
        } catch (Exception e) {
            log.error("云MAS业务平台短信验证码发送失败: {}", e.getMessage(), e);
            return false;
        }

    }

    @Override
    public boolean sendMasSmsContent(String phone, String content) {
        try {

            log.info("云MAS业务平台 短信内容: {}", content);
            log.info("云MAS业务平台 发送手机号: {}，短信内容：{}", phone, content);
            // 生成 MAC 签名
            String mac = SmsUtil.calculateMac(smsConfig.getEcName(), smsConfig.getApId(), smsConfig.getSecretKey(),
                    phone, content, smsConfig.getSign(), smsConfig.getAddSerial());
            System.out.println("云MAS业务平台 mac签名: " + mac);
            log.info("云MAS业务平台 mac签名: {}", mac);

            // 构建请求体并转换为 Base64 编码的 JSON 字符串
            String encodedJson = SmsUtil.encodeRequestBodyToBase64(buildRequestBody(phone, content, mac));
            System.out.println("云MAS业务平台 请求体Base64编码: " + encodedJson);
            log.info("云MAS业务平台 请求体Base64编码: {}", encodedJson);

            // 设置请求头并发送 POST 请求
            String response = SmsHttpUtil.sendPostRequest(smsConfig.getApiUrl(), encodedJson);
            System.out.println("云MAS业务平台 响应体: " + response);
            log.info("云MAS业务平台 响应体response: {}", response);

            // 解析响应并处理结果
            return SmsUtil.handleResponse(response);

        } catch (Exception e) {
            log.error("云MAS业务平台短信发送失败: {}", e.getMessage(), e);
            return false;
        }
    }

    @Override
    public void asyncSendMasSmsContent(String phone, String content){
        CompletableFuture.runAsync(() -> {
                    try {
                        System.out.println("======开始发送短信======");
                        sendMasSmsContent(phone, content);
                        System.out.println("======发送短信结束======");
                    } catch (Exception e) {
                        log.error("asyncSendMasSmsContent异步执行失败", e);
                    }
                }
        );
    }

    /**
     * 构建请求体
     */
    private Map<String, String> buildRequestBody(String phone, String content, String mac) {
        Map<String, String> requestBody = new HashMap<>();
        requestBody.put("ecName", smsConfig.getEcName());
        requestBody.put("apId", smsConfig.getApId());
        requestBody.put("mobiles", phone);
        requestBody.put("content", content);
        requestBody.put("secretKey", smsConfig.getSecretKey());
        requestBody.put("sign", smsConfig.getSign());
        requestBody.put("addSerial", smsConfig.getAddSerial());
        requestBody.put("mac", mac);
        return requestBody;
    }



}
