package com.cnooc.expert.common.utils;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.kafka.core.KafkaTemplate;
import org.springframework.kafka.support.SendResult;
import org.springframework.stereotype.Component;
import org.springframework.util.concurrent.ListenableFuture;
import org.springframework.util.concurrent.ListenableFutureCallback;

@Component
public class KafkaProducerUtil {

    @Autowired
    private KafkaTemplate<String, Object> kafkaTemplate;

    /**
     * 发送消息到指定主题
     * @param topic 主题名称
     * @param message 消息内容
     */
    public void sendMessage(String topic, Object message) {
        kafkaTemplate.send(topic, message);
    }

    /**
     * 发送消息到指定主题（带key）
     * @param topic 主题名称
     * @param key 消息key
     * @param message 消息内容
     */
    public void sendMessage(String topic, String key, Object message) {
        kafkaTemplate.send(topic, key, message);
    }

    /**
     * 发送消息到指定主题（带分区和key）
     * @param topic 主题名称
     * @param partition 分区
     * @param key 消息key
     * @param message 消息内容
     */
    public void sendMessage(String topic, Integer partition, String key, Object message) {
        kafkaTemplate.send(topic, partition, key, message);
    }

    /**
     * 异步发送消息，带回调函数
     * @param topic 主题名称
     * @param message 消息内容
     * @param callback 回调函数
     */
    public void sendMessageWithCallback(String topic, Object message,
                                        ListenableFutureCallback<SendResult<String, Object>> callback) {
        ListenableFuture<SendResult<String, Object>> future = kafkaTemplate.send(topic, message);
        future.addCallback(callback);
    }

    /**
     * 异步发送消息，带默认回调
     * @param topic 主题名称
     * @param message 消息内容
     */
    public void sendMessageWithCallback(String topic, Object message) {
        ListenableFuture<SendResult<String, Object>> future = kafkaTemplate.send(topic, message);
        future.addCallback(new ListenableFutureCallback<SendResult<String, Object>>() {
            @Override
            public void onSuccess(SendResult<String, Object> result) {
                System.out.println("发送消息成功: " + message + ", offset: " +
                        result.getRecordMetadata().offset());
            }

            @Override
            public void onFailure(Throwable ex) {
                System.err.println("发送消息失败: " + message + ", 错误: " + ex.getMessage());
            }
        });
    }
}